//------------------------------------------------------------
// Interactive Button Pad with SAM2695 Synth
// 2024.03.13 RSP
// Target: RP2040
//
// Games:
//    1) Ambient blinking
//    2) Interactive Game of Life
//    3) Recording Sequencer
//   Press button at boot for extended modes
//    4) lower 64 of main SAM2650 patches
//    5) upper 64 of main SAM2650 patches
//    6) lower 64 of MT-32 variant SAM2650 patches
//    7  upper 64 of MT-32 variant SAM2650 patches
//
// User interface:
//    Press any button on initial (randomization) screen
//    Pot controls speed
//    Encoder spin selects mode
//    Encoder press toggles sound
//    Encoder long press sets volume (spin encoder to set, press to exit)
//------------------------------------------------------------

//#define SERIAL_MONITOR  // enable console output

#define PWR_LED 17
#define POT_PIN A0

//-------------------------------------------------------------
// serial midi

#include <MIDI.h>
//#define MIDI_SERIAL Serial1
//MIDI_CREATE_INSTANCE(HardwareSerial, MIDI_SERIAL, MIDI); // RP2040 Rx=GP1, Tx=GP0
#define FIFOSIZE 256 // MIDI serial transmit/receive buffer size
SerialPIO SerialTx( 3, SerialPIO::NOPIN, FIFOSIZE );
MIDI_CREATE_INSTANCE(SerialPIO, SerialTx,  MIDI);

void stopAllNotes()
{
  for (uint ch=1; ch <= 11; ch++)
    MIDI.sendControlChange( 123,0,ch );
}

// collapse notes 0..79 to regular notes in octaves 
byte regularNote( byte n )
{
  const byte regular_notes[7] = { 0, 2, 4, 5, 7, 9,11 };
  uint oct = n/7;
  return oct*12 + regular_notes[n % 7];
}

const byte volume_table[8] = { 15,30,45,60,75,90,105,127 };

struct
{ byte variant;
  byte patch;
  byte pitch_base;
} patches[10] = { {0,0,30},{0,0,30},{0,0,30},{0,0,30},{0,0,30},{0,0,30},{0,0,30},{0,0,30},{0,0,30},{0,0,30} };

struct
{ byte variant;
  byte patch;
} crappy_patches[37] =
{ {0,13},{0,45},{0,47},{0,55},{0,101},{0,113},{0,115},{0,116},{0,117},{0,118},{0,118},{0,119},{0,120},{0,121},{0,122},{0,123},{0,124},{0,125},{0,126},{0,127},
  {1,38},{1,51},{1,57},{1,58},{1,111},{1,112},{1,113},{1,114},{1,115},{1,116},{1,117},{1,118},{1,119},{1,122},{1,123},{1,124},{1,125} };

void randomProgram( byte channel_index )
{
  byte pitch_base = random(15,45);
  for (;;)
  { byte variation = random(2);   // 0 or 1
    byte patch     = random(128);
    boolean blocked = false;
    // excluded patches
    for (uint i=0; i < 37; i++)
      blocked |= (crappy_patches[i].variant = variation && crappy_patches[i].patch == patch);
    if (blocked) continue;
    // don't reuse same patch
    for (uint i=0; i < 10; i++)
      blocked |= (patches[i].variant == variation && patches[i].patch == patch);
    if (blocked) continue;
    // assign patch to channel
    patches[channel_index].variant = variation;
    patches[channel_index].patch   = patch;
    patches[channel_index].pitch_base = pitch_base;
    byte ch = channel_index+1;  if (ch == 10) ch = 11;
    MIDI.sendControlChange( 0, variation ? 127:0, ch ); // ctrl 0 is bank select
    MIDI.sendProgramChange( patch, ch );
    break;
  }
}

void randomProgramAll()
{
  for (uint i=0; i < 10; i++) randomProgram(i);
debugPatches();
}

void debugPatches()
{
#ifdef SERIAL_MONITOR
Serial.println("patch map");
for (uint i=0; i < 10; i++) { Serial.print(patches[i].variant); Serial.print(","); Serial.println(patches[i].patch); }
#endif
}

//-------------------------------------------------------------
// note player

#include "player.h"
autoPlayer player;

//-------------------------------------------------------------
// button pad array

#include "VelociBus_4X4BP.h"
#include "vbus.h" // VelociBus superclass to handle 2x2 board arrangement
#define VBUS_SERIAL Serial2
VelociBus_2x2_boards vbus; // create VelociBus interface

//------------------------------------------------------
// knob

#include "EncoderSW.h"
EncoderSW knob;
#define BTN_PIN 22

#define NUM_MODES 3
#define NUM_EXTENDED_MODES 4
boolean enable_extended_modes;

//------------------------------------------------------
// 27C256 EEPROM
//
//    32768 bytes
//       64 byte page
//      512 pages

#define EE_CHIP EE_24C256
#include "fearless_24C_eeprom.h"

#define SETTINGS_UPDATE_MSEC 5000
boolean config_dirty = false;
ulong   config_tm;

#define EEPROM_SIGNATURE 0x3573
#define EEPROM_BLOCK_START  64   // block index 0
#define EEPROM_BLOCK_COUNT 511
#define EEPROM_PAGESIZE     64
struct EEPROM_block
{ byte polarity;
  byte volume;    // 0..7
  byte sound_on;
  byte mode;
};
EEPROM_block config;

void initializeEEPROM()
{
  // initialize new EEPROM
#ifdef SERIAL_MONITOR
Serial.println("NEW EEPROM");
#endif
  for (uint i=0; i < EEPROM_BLOCK_COUNT; i++)
  { if (EEwrite8( 0, EEPROM_BLOCK_START+i*EEPROM_PAGESIZE, 0 ))
      for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); }
    delay(1);
  }
}

uint locateCurrentEEblock() // returns block index
{
#ifdef SERIAL_MONITOR
Serial.print("locating config... ");
 #endif
 int pol = EEread8( 0, EEPROM_BLOCK_START );
  if (pol < 0) for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); }
  uint index;
  for (index=1; index < EEPROM_BLOCK_COUNT; index++)
  { int d = EEread8( 0, EEPROM_BLOCK_START+EEPROM_PAGESIZE*index );
    if (d < 0) for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); }
    if (d != pol) { Serial.println(index-1); return index-1; }
  }
#ifdef SERIAL_MONITOR
Serial.println("0");
#endif
  return 0; // wrapped
}

void writeConfig( uint index )
{
#ifdef SERIAL_MONITOR
Serial.print("writeConfig "); Serial.println(index);  
#endif
  if (index >= EEPROM_BLOCK_COUNT) 
  { index = 0; // handle wrap
    config.polarity = config.polarity ? 0:1;
  }
  if (EEwriteBlock( 0, EEPROM_BLOCK_START+index*EEPROM_PAGESIZE, (byte *)&config, 4 ))
      for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); }
}

boolean loadConfig()
{
  // check for new EEPROM
  word sig;
  if (EEreadBlock( 0, 0, (byte *)&sig, 2 ))
    for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); }
  if (sig != EEPROM_SIGNATURE) return false;
  // locate current data
  uint index = locateCurrentEEblock();
  if (EEreadBlock( 0, EEPROM_BLOCK_START+EEPROM_PAGESIZE*index, (byte *)&config, 4 ))
    for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); delay(100); }
  config.volume &= 7;
  if (config.mode >= NUM_MODES+NUM_EXTENDED_MODES) config.mode = 0;
  return true;
}

//------------------------------------------------------
// text

extern const unsigned char alpha_charset[5*59];

void paintChar( char c, uint x )
{
  if (c < '!' || c > 'z') return;
  uint cp = c-' '; // charset starts at space
       cp *= 5;
  for (uint i=0; i < 5; i++, cp++)
    for (uint r=0; r < 8; r++)
      if (alpha_charset[cp] & (0x80>>r))
        vbus.setXY( x+i, r, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::red );
}

//-------------------------------------------------------------
// interactive game of life

#include "life.h"
CONWAYS_LIFE game;
boolean restart = false;
uint    tempo   = 1000;

boolean  start_new_game = true; // start new game
VelociBus_4X4BP::color_code color = VelociBus_4X4BP::black;

boolean manual = false;
ulong   manual_tm;
#define MANUAL_MSEC 2000

boolean alldead;
ulong   alldead_tm;
#define RESTART_MSEC   5000 // after inactivity
#define MAX_LIFE_MSEC 60000 // max any one game

void lifeInit()
{
  stopAllNotes();
  start_new_game = true;
}

void lifeRun()
{
  static ulong tick_tm = millis();
  static ulong hard_tm = millis();
  static byte instrument = 0;
  
  //-- start/run game
  if (start_new_game)
  {
    start_new_game = false;
    alldead       = false;
    vbus.setLEDall( VelociBus_4X4BP::BOARD_BROADCAST_ADDR, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::black );
    tick_tm       = millis();
    hard_tm       = millis();
    game.initial  = 15; 
    game.start(true); // wrap
    VelociBus_4X4BP::color_code prev_color = color;
    while (color == prev_color) color = (VelociBus_4X4BP::color_code)random(1,16);
  }
  else // drive the game at tempo rate
    if (manual)
    { if (millis()-manual_tm > MANUAL_MSEC)
      { manual  = false;
        tick_tm = millis()-1000;
      }
    }
    else
      if (millis() - tick_tm >= tempo)
      {
        tempo = map( analogRead(POT_PIN), 0,1023, 2500,400 );
        tick_tm = millis();
        game.run();
        // render game board to display buffer
        for (uint r=0; r < 8; r++)
        { for (uint c=0; c < 8; c++)
            vbus.setXY( c,r, VelociBus_4X4BP::FCN_COLOR, 
                        (game.board[r] & (1 << c)) ? color : VelociBus_4X4BP::black );
        }
        if (config.sound_on)
        { // population = velocity
          byte q1 = 0; for (uint i=0; i < 4; i++) for (uint j=0; j < 4; j++) if (game.board[i] & (1 << j)) q1++;
          byte q2 = 0; for (uint i=4; i < 4; i++) for (uint j=4; j < 8; j++) if (game.board[i] & (1 << j)) q2++;
          byte q3 = 0; for (uint i=0; i < 4; i++) for (uint j=0; j < 4; j++) if (game.board[i] & (1 << j)) q3++;
          byte q4 = 0; for (uint i=4; i < 8; i++) for (uint j=4; j < 8; j++) if (game.board[i] & (1 << j)) q4++;
#ifdef SERIAL_MONITOR
Serial.println(q1+q2+q3+q4);
#endif
          uint velocity = volume_table[config.volume];
               velocity *= map( q1+q2+q3+q4, 0,64, 50,200 );
               velocity /= 100;
               if (velocity > 127) velocity = 127;
          if (q1+q2+q3+q4)
          { byte p        = ((q1 & 0x0C) ? 8:0) + ((q2 & 0x0C) ? 4:0) + ((q3 & 0x0C) ? 2:0) + ((q4 & 0x0C) ? 1:0);
            byte ch       = instrument+1;
            byte pitch    = regularNote(p+patches[ch].pitch_base);
            ulong on_time = random( 500,map(tempo, 400,2500, 1000,5000 ));
#ifdef SERIAL_MONITOR
Serial.print(p); Serial.print(" ");
Serial.print(ch); Serial.print(" ");
Serial.print(pitch); Serial.print(" ");
Serial.println(velocity);
#endif
            instrument = (instrument+1) % 11;
          
            if (ch == 10) ch = 11;     
            player.play( ch, pitch, velocity, on_time );
          }
        }
      }

  // randomly select a new patch
#define NEWPATCH_MSEC 30000  
  if (config.sound_on)
  { static uint channel_index = 0;
    static ulong tm = millis();
    if (millis() - tm > NEWPATCH_MSEC)
    { tm = millis();
      randomProgram( channel_index );
      channel_index = (channel_index+1) % 10;
debugPatches();    
    }
  }
  
  //-- restart after a while of inactivity
  if (!alldead)
  { uint i;
    for (i=0; i < 8; i++)
      if (game.board[i]) break;
    if (i >= 8)
    { alldead    = true;
      alldead_tm = millis();
    }
  }
  else if (millis()-alldead_tm > RESTART_MSEC) start_new_game = true;

  //-- restart after hard time limit
  if (millis() - hard_tm > MAX_LIFE_MSEC)
    start_new_game = true;

  //-- handle button presses
  { VelociBus_4X4BP::button_info btn;
    if (vbus.getButton( &btn ))
    { VelociBus_2x2_boards::point p;
      p = vbus.getXY( btn.board_address, btn.button_index );
      switch (btn.event)
      {
        case VelociBus_4X4BP::BTN_PRESS:
          game.board[p.y] ^= (1 << p.x);
          vbus.setXY( p.x,p.y, VelociBus_4X4BP::FCN_COLOR, 
                      (game.board[p.y] & (1 << p.x)) ? color : VelociBus_4X4BP::black );
          manual    = true;
          manual_tm = millis();
          alldead   = false;
          break;
          
        case VelociBus_4X4BP::BTN_RELEASE:
          break;
    
        case VelociBus_4X4BP::BTN_HOLD:
          start_new_game = true;
      }
    }
  }
}

//-------------------------------------------------------------
// ambient blinking

#define FADEOUT_MSECS 1000

struct board_cell
{ VelociBus_4X4BP::color_code color;
  uint  state;
  uint32_t tm;
  uint32_t on_time;
};
struct board { board_cell cell[16]; };
board *bp; // dynamically allocated for actual number of button panels

void ambientBlinkInit()
{
  stopAllNotes();
  player.init();    
  for (uint b=0; b < vbus.board_count; b++)
    for (uint i=0; i < 16; i++) // 16 per board
      bp[b].cell[i].state = 0;
}

void ambientBlinkRun()
{
  static ulong frame_ms = random(10,250);
  static ulong tm = millis();
  if (millis()-tm > frame_ms)
  {  
    // light up new cell
    for (uint m=0; m < 25; m++) // random tries
    { // look for dark cell
      uint b = random(0,vbus.board_count);
      uint i = random(0,16);
      if (bp[b].cell[i].state) continue;
      // start cell
      bp[b].cell[i].color = (VelociBus_4X4BP::color_code)random(1,16);
      bp[b].cell[i].tm = millis();
      bp[b].cell[i].state = 1;
      bp[b].cell[i].on_time = random(1000,5000);
      vbus.setLED( b,i, VelociBus_4X4BP::FCN_COLOR, bp[b].cell[i].color );
      if (config.sound_on) 
      { byte ch       = b*2+(i % 4)+1;
        byte pitch    = regularNote(i+patches[ch].pitch_base);
        byte velocity = volume_table[config.volume];
//Serial.println(ch); 
        if (ch == 10) ch = 11;     
        player.play( ch, pitch, velocity, bp[b].cell[i].on_time );
      }
      break;
    }
    tm = millis();
    frame_ms = random(10, map(analogRead(POT_PIN), 0,1023, 2000,100) );
  }
  else // animate
    for (uint b=0; b < vbus.board_count; b++)
      for (uint i=0; i < 16; i++)
        switch (bp[b].cell[i].state)
        {
          case 1: // on
            if (millis()-bp[b].cell[i].tm > bp[b].cell[i].on_time)
            { bp[b].cell[i].state = 2;
              bp[b].cell[i].tm = millis();
              vbus.setLED( b,i, VelociBus_4X4BP::FCN_FADEOUT, bp[b].cell[i].color );
            }
            break;
            
          case 2: // fading
            if (millis()-bp[b].cell[i].tm > FADEOUT_MSECS)
              bp[b].cell[i].state = 0; // square should be dark
            break;
          
          case 0: // empty
          default: bp[b].cell[i].state = 0;
        } 

  // randomly select a new patch
#define NEWPATCH_MSEC 30000  
  if (config.sound_on) 
  { static uint channel_index = 0;
    static ulong tm = millis();
    if (millis() - tm > NEWPATCH_MSEC)
    { tm = millis();
      randomProgram( channel_index );
      channel_index = (channel_index+1) % 10;
debugPatches();    
    }
  } 
}

//-------------------------------------------------------------
// recorder
//
//    1 ch 1   1 2 3 4 5 6 7 8  pitch
//    2 ch 2
//    3 ch 3
//    4 ch 4
//    5 ch 5
//    6 ch 6
//    7 ch 10  9 10 11 12 13 14 15 16  percussion
//    8 ch 10  1 2  3  4  5  6  7  8

const VelociBus_4X4BP::color_code row_color[8] = 
{
  VelociBus_4X4BP::blue,
  VelociBus_4X4BP::cyan,
  VelociBus_4X4BP::palegreen,
  VelociBus_4X4BP::orange,
  VelociBus_4X4BP::green,
  VelociBus_4X4BP::yellow,
  VelociBus_4X4BP::red,
  VelociBus_4X4BP::red  
};

#include <forward_list>  // linked list storage for recording
#include <vector>        // dynamic array

struct note_event
{ 
  ulong start_tm; // offset from start
  ulong on_msecs;
  byte channel;   // 1 or 10
  byte pitch;
  byte x;
  byte y;
};
std::forward_list<note_event> notes;

struct active_note // notes being recorded
{
  std::forward_list<note_event>::iterator note_position;
};
std::vector<active_note> active_notes;

struct playing_note // playing notes (not including notes being recorded)
{
  std::forward_list<note_event>::iterator note_position;
};
std::vector<playing_note> playing_notes;

#define RECORDER_RELOOP_TIMEOUT 10000
boolean recording = false;
float   rec_tempo;
uint    rec_display_index = 0;
ulong   playback_tm; // starting millis() of playback cycle
ulong   last_activity_tm;
std::forward_list<note_event>::iterator playback_position; // last note that was played

void sequencerInit()
{
  stopAllNotes();
  // load instruments
  const struct
  { byte channel;
    byte variant;
    byte patch;       
  } instruments[6] =
  { {1,0,0},   // grand pianl
    {2,0,11},  // vibraphone
    {3,0,19},  // church organ 
    {4,0,26},  // jazz guitar 
    {5,0,41},  // violin 
    {6,0,56}   // trumpet 
  };
  for (uint i=0; i < 6; i++)
  { MIDI.sendControlChange( 0, instruments[i].variant ? 127:0, instruments[i].channel ); // ctrl 0 is bank select
    MIDI.sendProgramChange( instruments[i].patch, instruments[i].channel );
  }
  // clear recorder memory
  notes.clear();
  active_notes.clear();
  playing_notes.clear();
  playback_position = notes.before_begin();
  // start pre-record display
  recording = false;
  rec_display_index = 0;
  playback_tm = millis();
}

void sequencerRun()
{
  
  { static ulong tm = millis();
    if (millis() - tm > 50)
    { tm = millis();
      uint pot = analogRead(POT_PIN);
      if (pot < 512)
           rec_tempo = map(pot,   0,511,  200,100 ) / 100.0;
      else rec_tempo = map(pot, 512,1023, 100,50  ) / 100.0;
//Serial.println(rec_tempo);
    }
  }
  
//-- pre-record display

  if (!recording)
  {
    if (millis() - playback_tm > 250)
    {
      playback_tm = millis();

      for (uint r=0; r < 8; r++)
        vbus.setXY( rec_display_index, r, VelociBus_4X4BP::FCN_FADEOUT, row_color[r] );
      rec_display_index = (rec_display_index+1) % 8;
    }
  }

//-- auto reloop after inactivity

  if (recording)
    if (playing_notes.empty())
      if (millis() - last_activity_tm > RECORDER_RELOOP_TIMEOUT)
      {
        playback_tm = last_activity_tm = millis();
        playback_position = notes.before_begin();
      }

//-- playback

  if (playback_position != notes.end())
  {
    // start playing next note if it's time
    std::forward_list<note_event>::iterator next_np = playback_position;
    next_np++;
    if (next_np != notes.end())
      if (next_np->start_tm * rec_tempo <= millis()-playback_tm)
      { 
        // play note
        if (config.sound_on) MIDI.sendNoteOn( next_np->pitch, volume_table[config.volume], next_np->channel );
        vbus.setXY( next_np->x, next_np->y, VelociBus_4X4BP::FCN_COLOR, row_color[next_np->y] );
        // advance
        playing_notes.push_back( { next_np } );
        playback_position++;
        last_activity_tm = millis();
      }
    
    // age playing notes
    for (auto pn_it = playing_notes.begin(); pn_it != playing_notes.end(); ++pn_it) 
      if (millis()-(playback_tm + pn_it->note_position->start_tm*rec_tempo) >= pn_it->note_position->on_msecs*rec_tempo)
      { // end playing note
        MIDI.sendNoteOff(pn_it->note_position->pitch, 0, pn_it->note_position->channel);
        boolean leave_lamp_on = false; // keep lamp on if duplicate is under construction now
        for (auto an_it = active_notes.begin(); an_it != active_notes.end(); ++an_it) 
          if (an_it->note_position->x == pn_it->note_position->x && an_it->note_position->y == pn_it->note_position->y)
          { leave_lamp_on = true;
            break;
          }
        if (!leave_lamp_on) vbus.setXY( pn_it->note_position->x, pn_it->note_position->y, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::black );
        playing_notes.erase(pn_it--);
        last_activity_tm = millis();
      }
  }

//-- record new notes

  VelociBus_4X4BP::button_info btn;
  if (vbus.getButton( &btn ))
    if (config.sound_on) // don't record if sound is off
  { // decode button information
    VelociBus_2x2_boards::point p;
    p = vbus.getXY( btn.board_address, btn.button_index );
    // determine channel & pitch for button
    byte pit = p.x+35;
    byte ch  = p.y+1;  if (ch == 8) pit += 8;  if (ch >= 7) ch = 10;

    switch (btn.event)
    {
      case VelociBus_4X4BP::BTN_PRESS:
      { // don't allow note overplay
        if (!sequencerAlreadyPlayingNote(ch,pit))
        // start recording if necessary
        if (!recording)
        { recording = true;
          playback_tm = millis();
          playback_position = notes.before_begin();
          vbus.setLEDall( VelociBus_4X4BP::BOARD_BROADCAST_ADDR, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::black );
        }
        // begin note construction
        note_event ne = { (ulong)((millis() - playback_tm)/rec_tempo), 0, ch, pit, p.x, p.y };

        // insert chronologically
        std::forward_list<note_event>::iterator np = notes.insert_after( playback_position, ne );
       
        // track note under construction
        active_notes.push_back( { np } );

        // start playing note
        playback_position = np;
        MIDI.sendNoteOn( pit, volume_table[config.volume], ch );
        vbus.setXY( p.x, p.y, VelociBus_4X4BP::FCN_COLOR, row_color[p.y] );
        last_activity_tm = millis();
        break;
      }
      case VelociBus_4X4BP::BTN_RELEASE:
        // locate active note
        if (!active_notes.empty()) // must not dereference invalid iterator
          for (auto an_it = active_notes.begin(); an_it != active_notes.end(); ++an_it) 
            if (an_it->note_position->channel == ch && an_it->note_position->pitch == pit)
            { // finish note construction
              an_it->note_position->on_msecs = millis() - (playback_tm + rec_tempo*an_it->note_position->start_tm);
//Serial.print("on time = "); Serial.println( an_it->note_position->on_msecs);              
              an_it->note_position->on_msecs /= rec_tempo;
              active_notes.erase(an_it--);
              // stop playing note
              MIDI.sendNoteOff(pit, 0, ch);
              last_activity_tm = millis();
              boolean leave_lamp_on = false; // keep lamp on if duplicate is under playing now
              for (auto pn_it = playing_notes.begin(); pn_it != playing_notes.end(); ++pn_it) 
                if (p.x == pn_it->note_position->x && p.y == pn_it->note_position->y)
                { leave_lamp_on = true;
                  break;
                }
              if (!leave_lamp_on) vbus.setXY( p.x, p.y, VelociBus_4X4BP::FCN_FADEOUT, row_color[p.y] );
              break;
            }
      case VelociBus_4X4BP::BTN_HOLD:
        break;
    }
  }
}

boolean sequencerAlreadyPlayingNote(byte ch, byte pit)
{
  if (!playing_notes.empty()) // must not dereference invalid iterator
    for (auto pn_it = playing_notes.begin(); pn_it != playing_notes.end(); ++pn_it) 
      if (pn_it->note_position->pitch == pit && pn_it->note_position->channel == ch) return true;
  return false;
}

//-------------------------------------------------------------
// randomization

boolean randomRun()
{
#define RANDOM_STEP_MSEC 500
  static uint frame = 0;
  static ulong tm = millis();
  static uint color = 0;

  random();
    
  if (millis() - tm > RANDOM_STEP_MSEC)
  {
    tm = millis();
    switch (frame)
    {
      case 0:
        vbus.setCol( 0, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        vbus.setCol( 7, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        vbus.setRow( 0, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        vbus.setRow( 7, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        break;
      case 1:
        for (uint c=1; c <= 6; c++) 
        { vbus.setXY( c,1, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
          vbus.setXY( c,6, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        }
        for (uint r=2; r <= 5; r++)
        { vbus.setXY( 1,r, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
          vbus.setXY( 6,r, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        }
        break;                
      case 2:
        for (uint c=2; c <= 5; c++) 
        { vbus.setXY( c,2, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
          vbus.setXY( c,5, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        }
        for (uint r=3; r <= 4; r++)
        { vbus.setXY( 2,r, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
          vbus.setXY( 5,r, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        }
        break;                
      case 3:
        vbus.setXY( 3,3, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        vbus.setXY( 4,3, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        vbus.setXY( 3,4, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        vbus.setXY( 4,4, VelociBus_4X4BP::FCN_FADEOUT, (VelociBus_4X4BP::color_code) (color+3) );
        color = (color+1) % 13;
    }
    frame = (frame+1) % 4;
  }

  //-- handle button presses
  { VelociBus_4X4BP::button_info btn;
    if (vbus.getButton( &btn ))
    { vbus.setLEDall( VelociBus_4X4BP::BOARD_BROADCAST_ADDR, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::black );
      return true;   
    }
  }
  return false;
}

//-------------------------------------------------------------
// volume setting

boolean volume_btn_press;

void paintVolume( byte volume ) // 0..7
{
  const VelociBus_4X4BP::color_code color = VelociBus_4X4BP::yellow;
  vbus.setLEDall( VelociBus_4X4BP::BOARD_BROADCAST_ADDR, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::black );
  vbus.setRow( 7, VelociBus_4X4BP::FCN_COLOR, color );
  for (int i=1; i <= volume; i++)
  { vbus.setXY( 3,7-i, VelociBus_4X4BP::FCN_COLOR, color );
    vbus.setXY( 4,7-i, VelociBus_4X4BP::FCN_COLOR, color );
  }
  for (int i=0; i <= (volume-2)/2; i++)
  { vbus.setXY( 2,6-i, VelociBus_4X4BP::FCN_COLOR, color );
    vbus.setXY( 5,6-i, VelociBus_4X4BP::FCN_COLOR, color );
  }  
  if (volume >= 6)
  { vbus.setXY( 1,6, VelociBus_4X4BP::FCN_COLOR, color );
    vbus.setXY( 6,6, VelociBus_4X4BP::FCN_COLOR, color );
  }  
}

void setVolumePage()
{
  stopAllNotes();
  volume_btn_press = false;
  knob.setButtonHandler( 0, volumeButtonHandler, 0 );
  paintVolume( config.volume );

  for (;;)
  { knob.loop(); // triggers button handler callback
    if (volume_btn_press)
    { knob.setButtonHandler( 0, myButtonHandler, 0 );
      vbus.setLEDall( VelociBus_4X4BP::BOARD_BROADCAST_ADDR, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::black );
      config.sound_on = true;
      return;
    }
    int spin = knob.getSpin(0); // spin changes volume
    if (spin)
    { int n_volume = config.volume + (spin > 0 ? 1 : -1);
      if (n_volume < 0) n_volume = 0;
      if (n_volume > 7) n_volume = 7;
      config.volume = n_volume;
      paintVolume( config.volume );
      config_dirty = true;  config_tm = millis();
    }  
  }
}

void volumeButtonHandler([[maybe_unused]]uint8_t index, uint8_t event)
{
  if (event == EncoderSW::BTN_PRESS) volume_btn_press = true;
}

//-------------------------------------------------------------
// instrument demonstrator

void instrumentsRun( uint variation, uint bank ) // bank 0 or 1
{
  { VelociBus_4X4BP::button_info btn;
    if (vbus.getButton( &btn ))
    { VelociBus_2x2_boards::point p;
      p = vbus.getXY( btn.board_address, btn.button_index );
      byte patch = bank*64 + p.y*8 + p.x;
#ifdef SERIAL_MONITOR
Serial.println(patch);
#endif
      if (variation)
        // SAM2560 table_value = patch+1      
        MIDI.sendControlChange( 0, 127, 1 );
      else
        MIDI.sendControlChange( 0, 0, 1 );

      MIDI.sendProgramChange( patch, 1 );
      player.play( 1, 60, 100, 1000 );
    }
  }
}

//-------------------------------------------------------------
// Program start

void setup() 
{
  pinMode( PWR_LED, OUTPUT );  digitalWrite( PWR_LED, HIGH );
  
  // diagnostics
#ifdef SERIAL_MONITOR  
  Serial.begin(115200); // usb serial
  while (!Serial && (millis() <= 10000));  delay(10);
  Serial.println("Started!");
#endif

  // button module on VeliciBus
  VBUS_SERIAL.begin(38400);    // VelociBus is always 38400
  if (!vbus.begin( &VBUS_SERIAL ))
    for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); delay(100); }
  // turn off all button lamps
  vbus.setLEDall( VelociBus_4X4BP::BOARD_BROADCAST_ADDR, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::black );

  // config in EEPROM on I2C
  Wire.begin();
  Wire.beginTransmission(EEPROM_ADDR);
  if (Wire.endTransmission() != 0) 
    for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); delay(100); }
  if (!loadConfig())
  { // uninitialized EEPROM
    initializeEEPROM();
    config.volume   = 5;
    config.sound_on = true;
    config.mode     = 0;
    config.polarity = 1;
    writeConfig(0);
    // EEPROM should be valid now
    word sig = EEPROM_SIGNATURE;
    if (EEwriteBlock( 0, 0, (byte *)&sig, 2 ))
        for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); }
  }
  
  // rotary encoder + button on GPIO pins
  knob.begin();

  // ambient blinking
  //   create board image
  bp = new board[vbus.board_count];
    
  // start midi
  MIDI.begin(MIDI_CHANNEL_OMNI);
  stopAllNotes();

  enable_extended_modes = (digitalRead( BTN_PIN ) == LOW);

  // randomize
  for (;;) if (randomRun()) break;
  
  knob.setButtonHandler( 0, myButtonHandler, 0 );
  initMode();
}

//-------------------------------------------------------------
// Bigloop

void loop() 
{
  random(); // roll the dice

  // update settings
  if (config_dirty)
    if (millis()-config_tm > SETTINGS_UPDATE_MSEC)
    { uint index = locateCurrentEEblock();
      EEPROM_block old_config;
      if (EEreadBlock( 0, EEPROM_BLOCK_START+index*EEPROM_PAGESIZE, (byte *)&old_config, 4 ))
        for (;;) { digitalWrite(PWR_LED,!digitalRead(PWR_LED)); delay(100); }
      if (memcmp( &config, &old_config, 4 ))
      {
        writeConfig( locateCurrentEEblock()+1 );
      }
      config_dirty = false;
    }

  // run current mode
  switch (config.mode)
  { case 0: ambientBlinkRun(); break;
    case 1: lifeRun();         break; 
    case 2: sequencerRun();    break;
    case 3: instrumentsRun(0,0); break;
    case 4: instrumentsRun(0,1); break;
    case 5: instrumentsRun(1,0); break;
    case 6: instrumentsRun(1,1); break;
  }
  player.run();
  
  // mode knob
#define SPIN_TIMEOUT 1000 // start new mode after inactivity timeout
  { boolean spinning = false;
    ulong   spin_tm;
    boolean init_mode = false;
    do
    { knob.loop(); // triggers button handler callback
      
      int spin = knob.getSpin(0); // spin selects mode
      if (spin)
      { int nmode = config.mode + (spin > 0 ? 1 : -1);
        int nmodes = NUM_MODES+(enable_extended_modes ? NUM_EXTENDED_MODES:0);
        if (nmode < 0)       nmode = nmodes-1;
        if (nmode >= nmodes) nmode = 0;
        config.mode = nmode;
        vbus.setLEDall( VelociBus_4X4BP::BOARD_BROADCAST_ADDR, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::black );
        paintChar( '1'+config.mode, 1 );        
        spinning = true;
        spin_tm  = millis();
        init_mode= true;
        config_dirty = true;  config_tm = millis();
      }
      if (spinning)
        if (millis() - spin_tm > SPIN_TIMEOUT) spinning = false;
    } while (spinning);
    if (init_mode) initMode();
  }
}

void initMode()
{ vbus.setLEDall( VelociBus_4X4BP::BOARD_BROADCAST_ADDR, VelociBus_4X4BP::FCN_COLOR, VelociBus_4X4BP::black );
  randomProgramAll();
  switch (config.mode)
  { case 0: ambientBlinkInit(); break;
    case 1: lifeInit();         break;
    case 2: sequencerInit();    break;
  }
}

void myButtonHandler([[maybe_unused]]uint8_t index, uint8_t event)
{
  switch (event)
  {
    case EncoderSW::BTN_PRESS:
      config.sound_on = !config.sound_on;
      if (!config.sound_on) stopAllNotes();
      config_dirty = true;  config_tm = millis();
      break;
      
    case EncoderSW::BTN_HOLD:
      setVolumePage(); 
  }
}
