//-------------------------------------------------------------------
// Button Pad 4x4
// 2024.01.14 RSP
// Target: ATmega328PB 5V/8MHz INTOSC using Minicore
//            clock 8MHz internal
//            BOD 2.7V (default)
//            Variant: 328PB
//            Bootloader: No
//            LTO: disabled
//         PROGRAMMING:
//            SPI using USBASP
//            set tools|board settings
//            BURN BOOTLOADER to set fuses
//            then UPLOAD USING PROGRAMMER
// Diagnostics:
//    Hold UL button at power on.
//      button 1 = all colors
//      button 2 = test pattern
//      button 3 = test pattern
//      button 15 = all off
//      long press = set brightnes (0..7)
//      
// Communication Protocol:
//    VelociRotor Special Protocol
//    Serial 38400
//    Modules can be daisy chained (4 pin connector, last board loops back)
/*
  VelociRotor Special Protocol 
  ----------------------------
  Header packet
  1 m m b a a a a   sync B7, message type, broadcast | board address

      master -> board 
          if address == broadcast, process message & forward as-is
          if address == 0, process message else --address & forward
        special diagnostic handling
          if message type == diagnostic, ++data & forward

      board -> master
          ++address & forward

  Chain Diagnostic packet
  1 1 1 1 0 0 0 0   header
  0 n n n n n n n   each board increments count & forwards it on

  Button packet
  1 1 0 x a a a a   header
  0 e e e a a a a   button events, button index

  Config packet
  1 0 0 b a a a a   header
  0 e e e x b b b   button event disable bits, brightness 0-7

  Lamp packet
  1 0 1 b a a a a   header
  0 f f f a a a a   function, lamp index
  0 x d d c c c c   pattern, color index
    Function 
      0 set color from palette (16 colors)
      1 slow breathe
      2 fast breathe
      3 slow blink
      4 fast blink
      5 color wheel (c = starting index)
      6 fade out
      7 flash in
    Pattern
      0 individual LED
      1 row    (lamp index = row)
      2 column (lamp index = col)
      3 all    (lamp index ignored)
*/
//-------------------------------------------------------------------

// serial monitor @ 115200 - dev only, not provisioned on PCB designs
//#define DEBUG_COMS 

#define LED_RGB true // vs GRB

//-------------------------------------------------------------------

byte config = 3; // button event disables (all events enabled) & brightness (3=medium)

#define BTN_PRESS   0b01000000
#define BTN_RELEASE 0b00100000
#define BTN_HOLD    0b00010000

//-------------------------------------------------------------------

int8_t testmode = 0; // 1 = enabled, -1 = disabled

#define NUM_BUTTONS 16

#include <Adafruit_NeoPixel.h>
#define PIX_PIN    3
#define NUM_PIXELS NUM_BUTTONS
#if LED_RGB 
Adafruit_NeoPixel pixels(NUM_PIXELS, PIX_PIN, NEO_RGB + NEO_KHZ800);
#else 
Adafruit_NeoPixel pixels(NUM_PIXELS, PIX_PIN, NEO_GRB + NEO_KHZ800);
#endif

enum lamp_fcn
{ 
  FCN_COLOR     = 0,
  FCN_BREATHE_S = 1,
  FCN_BREATHE_F = 2,
  FCN_BLINK_S   = 3,
  FCN_BLINK_F   = 4,
  FCN_WHEEL     = 5,
  FCN_FADEOUT   = 6,
  FCN_FLASHIN   = 7
};
enum lamp_pattern
{
  PAT_SINGLE = 0,
  PAT_ROW    = 1,
  PAT_COL    = 2,
  PAT_ALL    = 3
};

struct
{ lamp_fcn     fcn;
  uint32_t     color;
  byte         timer;  // for fade out & flash in functions  
} pixel[NUM_BUTTONS];

const uint32_t COLOR_TABLE[16] PROGMEM  =
{ //  ggrrbb
    0x000000, //  0 black
    0x808080, //  1 gray
    0xFFFFFF, //  2 white
    0xFFFF00, //  3 yellow
    
    0x5AFF00, //  4 orange
    0x40FF60, //  5 flesh
    0x00FF93, //  6 pink
    0x00FF40, //  7 magenta

    0x00FF00, //  8 red
    0xFF6000, //  9 pale green
    0xFF0000, // 10 green
    0xFF0040, // 11 pea green

    0xFF00FF, // 12 cyan
    0xA010FF, // 13 light blue
    0x2020FF, // 14 purple
    0x0000FF, // 15 blue
};

const byte BRIGHTNESS_TABLE[8] PROGMEM = 
{
  20,
  30,
  40,
  50,
  60,
  70,
  80,
  90,
};

#define FLASHIN_TICKS 10 // 10 msec ticks
const byte FLASHIN_TABLE[8] PROGMEM =     
{
  40,
  60,
  80,
  100,
  120,
  140,
  160,
  180,
};

const byte BREATHE_TABLE[256] PROGMEM =
{ // =$F$1+(255-$F$1)*SIN(A1*PI()/256)
  // =10  +(255-10  )*SIN(x *PI()/256)
  10,
  13,
  16,
  19,
  22,
  25,
  28,
  31,
  34,
  37,
  40,
  43,
  46,
  49,
  52,
  55,
  58,
  61,
  64,
  67,
  70,
  72,
  75,
  78,
  81,
  84,
  87,
  90,
  93,
  95,
  98,
  101,
  104,
  107,
  109,
  112,
  115,
  117,
  120,
  123,
  125,
  128,
  131,
  133,
  136,
  139,
  141,
  144,
  146,
  149,
  151,
  154,
  156,
  158,
  161,
  163,
  165,
  168,
  170,
  172,
  175,
  177,
  179,
  181,
  183,
  185,
  187,
  190,
  192,
  194,
  196,
  197,
  199,
  201,
  203,
  205,
  207,
  209,
  210,
  212,
  214,
  215,
  217,
  219,
  220,
  222,
  223,
  225,
  226,
  227,
  229,
  230,
  231,
  233,
  234,
  235,
  236,
  237,
  239,
  240,
  241,
  242,
  243,
  244,
  244,
  245,
  246,
  247,
  248,
  248,
  249,
  250,
  250,
  251,
  251,
  252,
  252,
  253,
  253,
  254,
  254,
  254,
  254,
  255,
  255,
  255,
  255,
  255,
  255,
  255,
  255,
  255,
  255,
  255,
  254,
  254,
  254,
  254,
  253,
  253,
  252,
  252,
  251,
  251,
  250,
  250,
  249,
  248,
  248,
  247,
  246,
  245,
  244,
  244,
  243,
  242,
  241,
  240,
  239,
  237,
  236,
  235,
  234,
  233,
  231,
  230,
  229,
  227,
  226,
  225,
  223,
  222,
  220,
  219,
  217,
  215,
  214,
  212,
  210,
  209,
  207,
  205,
  203,
  201,
  199,
  197,
  196,
  194,
  192,
  190,
  187,
  185,
  183,
  181,
  179,
  177,
  175,
  172,
  170,
  168,
  165,
  163,
  161,
  158,
  156,
  154,
  151,
  149,
  146,
  144,
  141,
  139,
  136,
  133,
  131,
  128,
  125,
  123,
  120,
  117,
  115,
  112,
  109,
  107,
  104,
  101,
  98,
  95,
  93,
  90,
  87,
  84,
  81,
  78,
  75,
  72,
  70,
  67,
  64,
  61,
  58,
  55,
  52,
  49,
  46,
  43,
  40,
  37,
  34,
  31,
  28,
  25,
  22,
  19,
  16,
  13
};
 
const byte FADEOUT_TABLE[92] PROGMEM = // ~10 msec/step
{ // -1 + exp(x/17.65)
  255,
  243,
  229,
  217,
  205,
  193,
  183,
  172,
  163,
  154,
  145,
  137,
  130,
  122,
  116,
  109,
  103,
  97,
  92,
  87,
  82,
  77,
  73,
  69,
  65,
  62,
  58,
  55,
  52,
  49,
  46,
  44,
  41,
  39,
  37,
  34,
  33,
  31,
  29,
  27,
  26,
  24,
  23,
  22,
  20,
  19,
  18,
  17,
  16,
  15,
  14,
  13,
  13,
  12,
  11,
  10,
  10,
  9,
  9,
  8,
  8,
  7,
  7,
  6,
  6,
  5,
  5,
  5,
  4,
  4,
  4,
  4,
  3,
  3,
  3,
  3,
  2,
  2,
  2,
  2,
  2,
  2,
  1,
  1,
  1,
  1,
  1,
  1,
  1,
  1,
  1,
  0
};

void setLED( byte led_index, lamp_fcn fcn, lamp_pattern pat, byte data )
{
  uint32_t c;
  if (fcn == FCN_WHEEL)
       c = data; // wheel phase
  else c = pgm_read_dword_near( COLOR_TABLE + (data & 0x0F) ); // look up color

  switch (pat)
  { case PAT_SINGLE:
      setLEDx( led_index, fcn, c );
      break;
    case PAT_COL:
      for (uint8_t i=0; i < 4; i++) setLEDx( (led_index & 0x03) + i*4, fcn, c );
      break;
    case PAT_ROW:
      for (uint8_t i=0; i < 4; i++) setLEDx( 4*(led_index & 0x03) + i, fcn, c );
      break;
    case PAT_ALL:
      for (uint8_t i=0; i < 4; i++) for (uint8_t j=0; j < 4; j++) setLEDx( i*4+j, fcn, c );
  }
}
void setLEDx( byte led_index, lamp_fcn fcn, uint32_t data )
{  
  pixel[led_index].fcn   = fcn;
  pixel[led_index].color = data;
  pixel[led_index].timer = 0;
}

uint32_t colorWheel( byte index )
{
  uint32_t r,g,b;
  if (index < 85)
  { r = 255 - index * 3;
    g = 0;
    b = index * 3;
  }
  else if (index < 170)
  {
    index -= 85;
    r = 0;
    g = index * 3;
    b = 255 - index * 3;
  }
  else
  { index -= 170;
    r = index * 3;
    g = 255 - index * 3;
    b = 0;
  }
  return (r << 16) + (g << 8) + b;
}

//-------------------------------------------------------------------
// transmit queue

#define Q_SIZE 512
uint16_t q_head = 0;
uint16_t q_tail = 0;
byte     q_data[ Q_SIZE ];

void quePost( byte header, byte data1, byte data2=0 )
{
  // if there's space in the queue
  uint16_t bytes_needed = ((header & 0b01100000) == 0b00100000) ? 3 : 2;  // lamp command is 3 bytes
  uint16_t bytes_used = (q_tail+Q_SIZE - q_head) % Q_SIZE;
  if ((Q_SIZE-1) - bytes_used < bytes_needed) 
  {
#ifdef DEBUG_COMS
Serial.println(F("TX QUE OVERFLOW!"));
#endif
    return; // oops, queue full, message lost
  }
  // post to queue
#ifdef DEBUG_COMS
Serial.print(F("POST: ")); Serial.print(header,HEX); Serial.print(" "); Serial.print(data1,HEX); 
if (bytes_needed == 3) { Serial.print(" "); Serial.print(data2,HEX); }
Serial.println();
#endif
  q_data[ q_tail ] = header;  q_tail = (q_tail+1) % Q_SIZE;
  q_data[ q_tail ] = data1;   q_tail = (q_tail+1) % Q_SIZE;
  if (bytes_needed == 3) { q_data[ q_tail ] = data2;  q_tail = (q_tail+1) % Q_SIZE; }
}

//-------------------------------------------------------------------
// buttons

// pushbutton state (one pushbutton)
struct pushbutton_t
{ boolean       state;      // follows button
  unsigned long tm;         // debounce timer
  boolean       hold_sent;
  unsigned long hold_tm;
};
#define DEBOUNCE_MS       50UL // contact debounce time
#define LONG_PRESS_MSEC 1000UL // long press time

pushbutton_t button [NUM_BUTTONS];

void handleButton( uint8_t button_index, boolean n_pressed )
{
  if (n_pressed) // button is released
  {
    if (!button[ button_index ].state)
      if (millis() - button[ button_index ].tm > DEBOUNCE_MS)
      { button[ button_index ].state = true;
        // send packet
        if ((config & BTN_RELEASE) == 0) // event enabled?
        {
#ifdef DEBUG_COMS
Serial.print(F("btn ")); Serial.print(button_index); Serial.println(F(" released")); 
#endif
          quePost( 0b11000000, BTN_RELEASE + button_index ); // board address is always zero
        }
    }
  }
  else // button is pressed
    if (button[ button_index ].state) // not already pressed?
    { // press event detected
      button[ button_index ].state = false;
      button[ button_index ].tm = millis(); // start debounce
      button[ button_index ].hold_tm = millis();
      button[ button_index ].hold_sent = false;

      // diagnostics: press selects a test pattern 
      if (testmode == 1)
      { 
       if (button_index == 0)
       {
         for (byte i=0; i < 16; i++) setLED( i, FCN_COLOR, PAT_SINGLE, i );
       }
       else if (button_index == 1)
       {
         setLED( 0, FCN_COLOR,     PAT_COL, random(15)+1 );
         setLED( 1, FCN_BREATHE_S, PAT_COL, random(15)+1 );
         setLED( 2, FCN_BREATHE_F, PAT_COL, random(15)+1 );
         setLED( 3, FCN_WHEEL,     PAT_COL, random(16) );
       }
       else if (button_index == 2)
       {
         setLED( 0, FCN_BLINK_S, PAT_COL, random(15)+1 );
         setLED( 1, FCN_BLINK_F, PAT_COL, random(15)+1 );
         setLED( 2, FCN_FADEOUT, PAT_COL, random(15)+1 );
         setLED( 3, FCN_FLASHIN, PAT_COL, random(15)+1 );
       }
       else if (button_index == 15)
       {
         setLED( 0, FCN_COLOR, PAT_ALL, 0 );
       }
       else
         setLED( button_index, FCN_COLOR, PAT_SINGLE, random(15)+1 );
      }

      // send packet
      if ((config & BTN_PRESS) == 0) // event enabled?
      {
#ifdef DEBUG_COMS
Serial.print(F("btn ")); Serial.print(button_index); Serial.println(F(" pressed")); 
#endif      
        quePost( 0b11000000, BTN_PRESS + button_index ); // board address is always zero
      }
    }
    else // button continues to be pressed
    { button[ button_index ].tm = millis(); // restart bounce
      // long press (hold) logic
      if (!button[ button_index ].hold_sent)
        if (millis() - button[ button_index ].hold_tm >= LONG_PRESS_MSEC)
        { button[ button_index ].hold_sent = true;

          // diagnostics: long press sets brightness
          if (testmode == 1)
            config = (config & 0xF8) + (button_index & 7);

          // send packet
          if ((config & BTN_HOLD) == 0) // event enabled?
          {
#ifdef DEBUG_COMS
Serial.print(F("btn ")); Serial.print(button_index); Serial.println(F(" hold")); 
#endif      
            quePost( 0b11000000, BTN_HOLD + button_index ); // board address is always zero
          }
        }
    }
}

//-------------------------------------------------------------------

void setup() 
{
  // column pullups
  PORTC |= 0b00111111; 

  // pull up unused pins
  PORTB |= 0b11110111; 
 
  // row selectors
  DDRE  = 0b00001111; 
  PORTE = 0b00001110; // preselect row 1

  // pixel output
  DDRD  = 0b00001000;
  PORTD = 0b11110111; // pull up unused pins
    
#ifdef DEBUG_COMS
  Serial.begin( 115200 ); // serial monitor
#endif

  Serial1.begin( 38400 ); // VelociBus

  // turn off neopixels
  pixels.begin(); // INITIALIZE NeoPixel strip object (REQUIRED)
  pixels.clear(); // Set all pixel colors to 'off'
  delay(10);
  pixels.show();  // Send the updated pixel colors to the hardware

  for (uint8_t i=0; i < 16; i++)
  { button[i].state = true;
    pixel[i].fcn   = FCN_COLOR; // solid black
    pixel[i].color = 0;
  } 
}

//-------------------------------------------------------------------

// check packet address
boolean isForMe( byte header )
{
  // it's for this board if address == 0 || broadcast bit is set
  return (((header & 0b1111) == 0) || (header & 0b00010000));
}

// forward packet if appropriate
void forward( byte header, byte data1, byte data2 = 0 )
{
  // forward broadcast | messages not for me
  if (header & 0b00010000) // broadcast ?
  { // forward packet
    quePost( header, data1, data2 );
  }
  else if (header & 0b1111) // not for this board ?
  { header--;
    quePost( header, data1, data2 );
  } 
}           

//- - - - - - - - - - - -

void loop() 
{

//-- 1 msec tick

  { static uint32_t last_ms = millis();
    if (last_ms != millis())
    { last_ms = millis();

//-- debounce buttons

      const byte row_select[4] = { 0b1101, 0b1011, 0b0111, 0b1110 };
      // scan buttons
      for (byte r=0; r < 4; r++)
      { // read columns
        byte b = PINC;
        // select next row, let it settle while processing the columns
        PORTE = row_select[r];
        // process a column of buttons
        for (byte c=0; c < 4; c++) handleButton( r*4 + c, b & (1 << c) );
        // diagnostics
        if (testmode == 0)
        { if ((b & 0x1) == 0) // first button pressed ?
            testmode = 1; else testmode = -1;
        }
      }

//-- refresh the LEDs

#define HILITE_COLOR 0xFFFFFF // solid color while pressed

#define SLOWBREATHE_PERIOD 250 // x10 msec
#define SLOWBREATHE_INCR (256UL*256UL/SLOWBREATHE_PERIOD)
#define FASTBREATHE_PERIOD 100 // x10 msec
#define FASTBREATHE_INCR (256UL*256UL/FASTBREATHE_PERIOD)

#define SLOWBLINK_PERIOD 100 // x10 msec
#define SLOWBLINK_ONTIME  75 // x10 msec
#define FASTBLINK_PERIOD  50 // x10 msec
#define FASTBLINK_ONTIME  25 // x10 msec
#define WHEEL_SPEED        2 // bigger is slower

union fracTimer
{ uint16_t acc;
  struct { uint8_t frac; uint8_t whole; } value;
};

      { static uint8_t slowblink_timer = 0;
        static uint8_t fastblink_timer = 0;
        
        static uint8_t wheel_index     = 0;
        static uint8_t wheel_divider   = 0;
               
        static fracTimer slowbreathe_timer;
        static fracTimer fastbreathe_timer;
        
        static byte tick = 0;
        if (++tick > 9) // every 10 msec
        { tick = 0;

          // maintain timers
          slowbreathe_timer.acc += SLOWBREATHE_INCR;
          fastbreathe_timer.acc += FASTBREATHE_INCR;
          slowblink_timer = (slowblink_timer+1) % SLOWBLINK_PERIOD;
          fastblink_timer = (fastblink_timer+1) % FASTBLINK_PERIOD;
          if (++wheel_divider >= WHEEL_SPEED)
          { wheel_divider = 0;
            wheel_index++;
          }

#define APPLY_BRIGHTNESS(mult) { for (uint8_t i=0; i < 3; i++, p++) *p = ((uint16_t)mult * (*p)) >> 8; }
          
          // populate pixels[] from pixel[] data
          uint8_t bm = pgm_read_byte_near( BRIGHTNESS_TABLE + (config & 0x07) ); 
          uint8_t *p = pixels.getPixels();
          for (uint8_t i=0; i < NUM_BUTTONS; i++)

            if (!button[i].state) // is button pressed now ?
            { // hilite button
              pixels.setPixelColor( i, HILITE_COLOR );
              byte m = pgm_read_byte_near( FLASHIN_TABLE + (config & 0x07) );
              APPLY_BRIGHTNESS(m); // apply brightness multiplier to this pixel
            }
            else // normal animation
            {
              switch (pixel[i].fcn)
              {
                case FCN_COLOR: // solid color
                  pixels.setPixelColor( i, pixel[i].color );
                  APPLY_BRIGHTNESS(bm); // apply brightness multiplier to this pixel
                  break;
               
                case FCN_BREATHE_S: // slow breathe
                { pixels.setPixelColor( i, pixel[i].color );
                  byte m = pgm_read_byte_near( BREATHE_TABLE + slowbreathe_timer.value.whole );
                  m = ((uint16_t)m * (uint16_t)bm) >> 8; // combine brightness & fade multipliers
                  APPLY_BRIGHTNESS(m); // apply brightness multiplier to this pixel
                  break;
                }
                case FCN_BREATHE_F: // fast breathe
                { pixels.setPixelColor( i, pixel[i].color );
                  byte m = pgm_read_byte_near( BREATHE_TABLE + fastbreathe_timer.value.whole );
                  m = ((uint16_t)m * (uint16_t)bm) >> 8; // combine brightness & fade multipliers
                  APPLY_BRIGHTNESS(m); // apply brightness multiplier to this pixel
                  break;
                }
                
                case FCN_BLINK_S: // slow blink
                { uint32_t c = 0;
                  if (slowblink_timer < SLOWBLINK_ONTIME) c = pixel[i].color;
                  pixels.setPixelColor( i, c );
                  APPLY_BRIGHTNESS(bm); // apply brightness multiplier to this pixel
                  break;
                }
                case FCN_BLINK_F: // fast blink
                { uint32_t c = 0;
                  if (fastblink_timer < FASTBLINK_ONTIME) c = pixel[i].color;
                  pixels.setPixelColor( i, c );
                  APPLY_BRIGHTNESS(bm); // apply brightness multiplier to this pixel
                  break;
                }
                
                case FCN_WHEEL: // color wheel (d = starting index)
                  pixels.setPixelColor( i, colorWheel(wheel_index + (pixel[i].color << 4)) );
                  APPLY_BRIGHTNESS(bm); // apply brightness multiplier to this pixel
                  break;

                case FCN_FADEOUT: // slow fade out
                { if (pixel[i].timer >= sizeof(FADEOUT_TABLE)) break; // failsafe
                  byte m = pgm_read_byte_near( FADEOUT_TABLE + pixel[i].timer );
                  if (m)
                  { m = ((uint16_t)m * (uint16_t)bm) >> 8; // combine brightness & fade multipliers
                    pixels.setPixelColor( i, pixel[i].color );
                    APPLY_BRIGHTNESS(m); // apply brightness multiplier to this pixel
                    if (pixel[i].timer < sizeof(FADEOUT_TABLE)-1) pixel[i].timer++;
                  }
                  else // fadeout ended
                  { pixel[i].color = 0;
                    pixel[i].fcn   = FCN_COLOR;
                    pixels.setPixelColor( i, pixel[i].color );
                    p += 3;
                  }
                  break;
                }

                case FCN_FLASHIN: // initial bright, then normal intensity
                { pixels.setPixelColor( i, pixel[i].color );
                  if (pixel[i].timer < FLASHIN_TICKS)
                  { // use alternate brightness for flash
                    byte m = pgm_read_byte_near( FLASHIN_TABLE + (config & 0x07F) );
                    APPLY_BRIGHTNESS(m); // apply brightness multiplier to this pixel
                    pixel[i].timer++;
                  }
                  else // flashin ended
                  { pixel[i].fcn   = FCN_COLOR;
                    APPLY_BRIGHTNESS(bm); // apply brightness multiplier to this pixel
                  }
                  break;
                }
              }
            }

          // send to LED string
          //   this takes about 0.25 msec
          pixels.show();
        }
      }
    }
  }

//-- listen for comms
 
  // serial monitor diagnostic console
#ifdef DEBUG_COMS
  {
    int c = Serial.read();
    switch (c)
    {
      case '0':
        setLED( 0, FCN_COLOR, PAT_ALL, 0 );
        break;
              
      case '1':
        setLED( 0, FCN_COLOR, PAT_ALL, 2 );
        break;

      case '2':
        setLED( 1, FCN_BLINK_F, PAT_ROW, 3 );
        break;

      case '3':
        setLED( 2, FCN_BLINK_S, PAT_COL, 4 );
        break;

      case '4':
        setLED( 9, FCN_BREATHE_F, PAT_ALL, 5 );
        break;

      case '5':
        setLED( 10, FCN_BREATHE_S, PAT_SINGLE, 6 );
        break;

      case '6':
        setLED( 11, FCN_WHEEL, PAT_SINGLE, 0 );
        break;

      case '7':
        setLED( 12, FCN_WHEEL, PAT_SINGLE, 32 );
        break;

      case '8':
        setLED( 14, FCN_FADEOUT, PAT_SINGLE, 2 );
        break;

      case '9':
        setLED( 13, FCN_FLASHIN, PAT_SINGLE, 3 );
        break;
    }
  }
#endif

  // listen for VelociBus packets
  { 
    static byte header = 0;
    static byte data1;
    static byte count;
    static uint32_t tm;

    int c = Serial1.read();
    if (c != -1)
    { if (c & 0x80) // header byte?
      { header = c; // save for later
        count  = 0; // no data bytes yet
        tm = millis();
      }
      else // possible data byte
      {
        if (header) // got a header byte?
          switch (header & 0b01100000)
          { 
            case 0b00000000: // config
#ifdef DEBUG_COMS
Serial.print(F("PKT(cfg) ")); Serial.print(header,HEX); Serial.print(" "); Serial.println(c,HEX);            
#endif
              if (isForMe(header)) config = c; // process packet
              forward(header,c); // forward packet if appropriate
              header = 0;
              break;
              
            case 0b00100000: // lamp control
              if (count == 0)
              { data1 = c;
                count++;
              }
              else // got both data bytes
              { 
#ifdef DEBUG_COMS
Serial.print(F("PKT(lamp) ")); Serial.print(header,HEX); Serial.print(" "); Serial.print(data1,HEX);  Serial.print(" "); Serial.println(c,HEX);            
#endif
                if (isForMe(header)) 
                { // process packet
                  byte     addr    = data1 & 0b1111;
                  lamp_fcn fcn     = (lamp_fcn)(data1 >> 4);
                  lamp_pattern pat = (lamp_pattern)(c >> 4);
                  setLED( addr, fcn, pat, c );
                }
                forward(header,data1,c); // forward packet if appropriate
                header = 0;
              }
              break;

            case 0b01000000: // button event
#ifdef DEBUG_COMS
Serial.print(F("PKT(btn) ")); Serial.print(header,HEX); Serial.print(" "); Serial.println(c,HEX);            
#endif
              // increment address & forward
              header++;
              quePost( header,c );
              header = 0;
              break;
                        
            case 0b01100000: // diagnostic
#ifdef DEBUG_COMS
Serial.print(F("PKT(diag) ")); Serial.print(header,HEX); Serial.print(" "); Serial.println(c);            
#endif
              // increment data & forward
              c++;
              forward(header,c);
          }
      }
    }
    if (millis() - tm > 50) header = 0; // time out invalid packets
  }

//-- spool the transmit queue out

  if (q_tail != q_head)
  { // transmit next message
    if ((q_data[ q_head ] & 0b10000000) == 0) // sanity check on header
    {
#ifdef DEBUG_COMS
Serial.println(F("RECOVERING QUEUE"));      
#endif
      q_head = q_tail = 0; // queue is fucked, reset it
    }
    else
    { uint8_t bytes = ((q_data[ q_head ] & 0b01100000) == 0b00100000) ? 3 : 2;
#ifdef DEBUG_COMS
Serial.print(F("Sending ")); Serial.println(bytes);    
#endif
      Serial1.write( q_data[ q_head ] );  q_head = (q_head+1) % Q_SIZE;
      Serial1.write( q_data[ q_head ] );  q_head = (q_head+1) % Q_SIZE;
      if (bytes == 3) { Serial1.write( q_data[ q_head ] );  q_head = (q_head+1) % Q_SIZE; }
    }    
  }

}
