//---------------------------------------------------------
// Piezo Touch Neopixel Fidget Toy
// 2022.04.13 RSP
// Target: Pro Mini 3.3V/8MHz (run at 5V) or 5V/16MHz
//
// Operation:
//    Touch to enter melody, starts playing after 5 seconds of inactivity
//    Touch to end playback
//    Auto mutes after 1 minute, touch to unmute
//---------------------------------------------------------

// piezo on pins 9 & 10 (as required by toneAC)
//   with 1M-10M load resistor in parallel
//   pin 10 is also joined to analog input A0
#include <toneAC.h>
#define PIEZO_PIN 10     // speaker output & analog input
#define AUTOMUTE_MS (1UL*60UL*1000UL) // 1 minute

// LED string
//   50-100 pixels is good
//   balance number of pixels with recording size to maximize RAM use
#include <FastLED.h>
#define LED_PIN     8
#define NUM_LEDS    72   // this app can handle up to about 100 LEDs
#define BRIGHTNESS  12   // 8 is low enough to run from USB
#define LED_TYPE    WS2811
#define COLOR_ORDER GRB
CRGB leds[NUM_LEDS];

#define FRAME_RATE_MS 20 // 50/sec refresh rate
#define MAX_SPRITES   25 // max sprites
#define MAX_RECORDS  175 // max sprites in playback recording

// live sprites plus a spawn sprite (#0)
struct sprite_t
{ boolean alive;
  uint8_t color8; // hue, random
  uint8_t length; // pixels
  uint8_t speed;  // velocity (tone)
  boolean leadin; // starting sprite vs exposed sprite
  union // generator for X = frame_rate * speed/256
  { struct { uint8_t frac_part; uint8_t int_part; };
    uint16_t accumulator;
  } X;
};  
sprite_t sprite[ MAX_SPRITES ];

// playback recording
struct record_t
{ uint16_t dt;    // msec from previous entry
  uint8_t color8;
  uint8_t length;
  uint8_t speed;
};
record_t record[ MAX_RECORDS ];
uint8_t num_records;   // number of records stored
// playback state
enum player_t { PLAYER_ARMED, PLAYER_RECORDING, PLAYER_PLAYBACK };
player_t player_state = PLAYER_ARMED;
uint8_t  player_pc;    // program counter
uint32_t player_tm;    // timing

// play real notes
const uint16_t FREQ_TABLE[32] =
{
  330, // E4
  349, // F4
  370, //  F#4/Gb4 
  392, // G4
  415, //  G#4/Ab4 
  440, // A4
  466, //  A#4/Bb4 
  494, // B4
  523, // C5
  554, //  C#5/Db5 
  587, // D5
  622, //  D#5/Eb5 
  659, // E5
  698, // F5
  740, //  F#5/Gb5 
  784, // G5
  831, //  G#5/Ab5 
  880, // A5
  932, //  A#5/Bb5 
  988, // B5
  1047, // C6
  1109, //  C#6/Db6 
  1175, // D6
  1245, //  D#6/Eb6 
  1319, // E6
  1397, // F6
  1480, //  F#6/Gb6 
  1568, // G6
  1661, //  G#6/Ab6 
  1760, // A6
  1865, //  A#6/Bb6 
  1976, // B6
};

//---------------------------------------------------------

void setup()
{
//Serial.begin(115200); // diagnostics

  // set up LED string
  FastLED.addLeds<LED_TYPE, LED_PIN, COLOR_ORDER>(leds, NUM_LEDS);
  FastLED.setBrightness(  BRIGHTNESS );

  // LED POST
  for (uint8_t i=0; i < NUM_LEDS; i++) leds[i] = CRGB::Red;    FastLED.show();  delay(1000);
  for (uint8_t i=0; i < NUM_LEDS; i++) leds[i] = CRGB::Green;  FastLED.show();  delay(1000);
  for (uint8_t i=0; i < NUM_LEDS; i++) leds[i] = CRGB::Blue;   FastLED.show();  delay(1000);
  for (uint8_t i=0; i < NUM_LEDS; i++) leds[i] = 0;            FastLED.show();

  // piezo is sensor plus speaker (not at the same time)
  toneAC( 1000, 10, 500 ); // hello world
  delay(10); // piezo stabilization delay
  pinMode( PIEZO_PIN, INPUT );
    
//Serial.println(F("started!"));
}

//---------------------------------------------------------
  
void loop() 
{
  random(); // spin the dice
  
//-- record player

  static uint32_t tone_tm = 0;   // tone duration tracking
  static uint16_t tone_duration;
  
  static uint32_t mute_tm;       // automatic mute timer

  if (player_state == PLAYER_PLAYBACK) // in playback mode?
  {
    // switch piezo to input 5ms after tone ends
    if (tone_tm != 0)
      if (millis() - tone_tm > tone_duration+5UL)
      { pinMode(PIEZO_PIN,INPUT);
        tone_tm = 0;
      }

    // run playback sequencer
    if (millis() - player_tm >= record[player_pc].dt) // time for next event ?
    { player_tm = millis();
      // find an unused sprite entry
      for (uint8_t i=1; i < MAX_SPRITES; i++) // exclude spawn sprite entry
        if (!sprite[i].alive)
        { // activate sprite from recording
          sprite[i].alive = true;
          sprite[i].color8 = record[player_pc].color8;        // random color
          sprite[i].length = record[player_pc].length;
          sprite[i].speed  = record[player_pc].speed;
          sprite[i].leadin = true;     
          sprite[i].X.accumulator = 0;
          uint16_t f = FREQ_TABLE[32-(sprite[i].speed >> 3)]; // keyboard velocity  => tone
          uint16_t t = map(sprite[i].length, 2,20, 200,500);  // keyboard hold time => sprite length
          if (millis() - mute_tm < AUTOMUTE_MS)
          { toneAC( f, 10, t, true ); // play tone in backgnd
            tone_tm = millis();       // track tone time in foregnd
            tone_duration = t;
//Serial.print(f); Serial.print(", "); Serial.println(t);          
          }
          break;
        }
      // advance playback index
      player_pc++;
      // automatically loop playback
      if (player_pc >= num_records) player_pc = 0;        
    }
  }
         
//-- frame refresh

  { static uint32_t keepalive_tm;   // recording timeout
    static uint32_t tm = millis();  // frame refresh timer
    if (millis() - tm >= FRAME_RATE_MS)
    { tm += FRAME_RATE_MS;

      // clear display buffer
      for (uint8_t i=0; i < NUM_LEDS; i++) leds[i] = 0;

//-- sprite spawner

      { // sprite for construction 
        //   keep display X locked at zero, use W to calculate length
        static union 
        { struct { uint8_t frac_part; uint8_t int_part; };
          uint16_t accumulator;
        } W;

        // recording timers for calculating time between events
        static uint32_t start_tm; 
        static uint32_t last_tm;

// piezo sensor input is 10 bit ADC
#define SPEED_DIVISOR 2 // 10 => 8 bits, no additional attenuation
        
        // read the ADC
        //   & downscale adc to 8 bits, plus any additional attenuation
        uint8_t s = analogRead(A0) >> SPEED_DIVISOR; // speed level

// piezo sense thresholds, where level is 0..255
#define TRIGGER_MIN 50  // level needed to start sprite
#define HOLD_MIN    25  // level needed to maintain sprite

        if (player_state == PLAYER_PLAYBACK) // playback active ?
        { // playback active
          //   touch controls playback: unmute or end playback
          if (tone_tm == 0) // detect touch when not playing a tone
            if (s >= TRIGGER_MIN)
            { if (millis() - mute_tm > AUTOMUTE_MS) // muted ?
                // unmute
                mute_tm = millis();
              else // end playback
              { FastLED.show(); // blank LEDs immediately
                for (uint8_t i=1; i < MAX_SPRITES; i++) sprite[i].alive = false;
                player_state = PLAYER_ARMED; // playback ended, ready to accept user input
//Serial.println("ARMED");              
              }
              // wait for touch release
              for (uint32_t tm=millis(); millis()-tm < 50UL; delay(1))
                if ((analogRead(A0) >> SPEED_DIVISOR) >= HOLD_MIN) tm=millis();
            }
        }
        else // armed or record mode, accepting touch input
        {
          if (sprite[0].alive) // building a sprite already ?
          { // building sprite
            if (s < HOLD_MIN)
            { // finish sprite & move it to the active sprites
              memmove( (void*)&sprite[1], (void*)&sprite[0], sizeof(sprite_t) * (MAX_SPRITES-1) );
              sprite[0].alive = false; // reset spawn sprite
              // store to recording buffer
              if (player_state == PLAYER_ARMED) // automatically start recording at 1'st input
              { num_records = 0; 
                player_state = PLAYER_RECORDING;
              }
              if (num_records < MAX_RECORDS) // failsafe
              { // delay 3 seconds at start of loop
                record[ num_records ].dt = (num_records == 0) ? 3000 : (start_tm-last_tm);
                record[ num_records ].color8 = sprite[0].color8;
                record[ num_records ].length = sprite[0].length;
                record[ num_records ].speed  = sprite[0].speed;
                num_records++;
                last_tm = start_tm;
//Serial.print(record[ num_records-1 ].speed); Serial.println(" REC");              
              }      
            }
            else // contine building sprite
            { keepalive_tm = millis();
              // speed is max speed seen
              if (s > sprite[0].speed) sprite[0].speed = s;
              // generate length at speed
              W.accumulator += (sprite[0].speed << 1); // generate at 2x normal speed
              sprite[0].length = W.int_part;
              // limit sprite length to LED string length
              if (sprite[0].length > NUM_LEDS) sprite[0].length = NUM_LEDS;
            }
          }
          else if (s >= TRIGGER_MIN)
          { // start sprite
            start_tm = millis();
            sprite[0].alive = true;
            sprite[0].color8 = random8();    //CHSV(random8(),255,255);
            sprite[0].length = 1;
            sprite[0].speed  = s;            // initial speed, can increase   
            sprite[0].X.accumulator = 0;
            sprite[0].leadin = false;        // sprite leadin is handled by generator
            W.int_part = 1; W.frac_part = 0; // start with length = 1
            keepalive_tm = millis();
          }
        
          // render spawn sprite
          if (sprite[0].alive)
          { CRGB c = CHSV(sprite[0].color8,255,255);
            for (uint8_t w=0; w < sprite[0].length; w++)
              leds[w] = c;
          }
        }
      }

      // start playback after 5 seconds of inactivity
      if (player_state == PLAYER_RECORDING && millis() - keepalive_tm >= 5000UL)
      { player_state = PLAYER_PLAYBACK; // set playback mode
        player_pc = 0;        // rewind
        player_tm = millis(); // initialize playback timers
        mute_tm   = millis();
      }

//-- sprite animator
      
      // scan all sprites except the spawn sprite (#0)
      for (uint8_t i=1; i < MAX_SPRITES; i++)
        if (sprite[i].alive)
        { // advance X
          sprite[i].X.accumulator += (sprite[i].speed << 0);
          if (sprite[i].leadin)
            if (sprite[i].X.int_part >= sprite[i].length) 
            { sprite[i].X.int_part = 0; sprite[i].leadin = false; }
          uint8_t px,pw;
          if (sprite[i].leadin) // sprite is starting up, partially hidden
          { px = 0;
            pw = sprite[i].X.int_part;
          }
          else // sprite is fully exposed
          { px = sprite[i].X.int_part;
            pw = sprite[i].length;
          } 
          // kill sprite when it goes off the end of the string       
          if (px >= NUM_LEDS) { sprite[i].alive = false; continue; }
          // render sprite       
          CRGB c = CHSV(sprite[i].color8,255,255);
          for (; pw; pw--, px++)
            if (px >= NUM_LEDS) break; else // failsafe
              leds[px] |= c; // color combine function is max of each color
        }

//-- send completed buffer to LEDs

      FastLED.show();
    }
  }
}
