//----------------------------------------------------------------
// Mega Fidget
// 2024.06.01 RSP
// Target: Mega 2560
//       + all parts from "ELEGOO Mega 2560 The Most Complete Starter Kit"
//----------------------------------------------------------------

#include <Wire.h>

// time warp based on speed knob
uint32_t frame_msec = 1000; // 1000 for real time
uint32_t my_millis  = 0;

// coroutine macros
#define coBegin { static int _state_ = 0; switch(_state_) { case 0:;
#define coEnd     _state_ = 0; }}
#define coWaitWhile(expr) { _state_ = __LINE__; return;  case __LINE__: if (expr) return; }
#define coDelay(msec) { static uint32_t tm; _state_ = __LINE__; tm=millis(); return; case __LINE__: if (millis()-tm < msec) return; }
#define coDelayWarped(msec) { static uint32_t tm; _state_ = __LINE__; tm=my_millis; return; case __LINE__: if (my_millis-tm < msec) return; }
#define coDebounce(expr,msec) { static uint32_t tm; _state_ = __LINE__; tm=millis(); return; case __LINE__: if (!(expr)) tm=millis(); if (millis()-tm < msec) return; }
// coroutines supplied with context
struct CoroutineContext
{ int      state;
  uint32_t tm; // for coDelayWarped
};
#define coBeginX(cc) { switch(cc.state) { case 0:;
#define coEndX(cc)     cc.state = 0; }}
#define coDelayWarpedX(cc,msec) { cc.state = __LINE__; cc.tm=my_millis; return; case __LINE__: if (my_millis-cc.tm < msec) return; }
#define coWaitWhileX(cc,expr) { cc.state = __LINE__; return;  case __LINE__: if (expr) return; }
#define coResetX(cc) cc.state=0;

//----------------------------------------------------------------
// Mega2560 timer interrupt
//   TimerInterrupt.h fights with Servo.h, so using PWMServo.h

#define USE_TIMER_1     true
#include "TimerInterrupt_Generic.h" // https://github.com/khoih-prog/TimerInterrupt_Generic
#define TIMER_INTERVAL_MS 1

//----------------------------------------------------------------
// analog inputs

#define POT_PIN   A0 // speed knob
#define JOY_Y_PIN A1
#define JOY_X_PIN A2 
#define RAIN_PIN  A3
#define SOUND_PIN A4
#define NTC_PIN   A5
#define BTN_PIN   A6
#define LDR1_PIN  A7
#define LDR2_PIN  A8

//----------------------------------------------------------------
// RFID

#define RST_PIN   11
#define SS_PIN    53

#include <SPI.h>
#include "MFRC522.h" // from Elegoo tutorial library
MFRC522 mfrc522(SS_PIN, RST_PIN); // Create MFRC522 instance
MFRC522::MIFARE_Key key;

//----------------------------------------------------------------
// tone player

#define TONE_PIN 38

struct melodyData
{
  uint16_t frequency; // Hz
  uint16_t duration;  // msec
};
const melodyData *tone_data = NULL;
uint16_t          tone_pc;
boolean           tone_active = false;

const melodyData test_melody[] = { {240,500},{350,500},{440,500} };

void tonePlayerTask() // run the melody player
{
  static uint32_t warped_duration;
  coBegin
    for (tone_pc=0; tone_data != NULL; tone_pc++)
    { if (tone_data[tone_pc].duration == 0) break; // end of melody marker
      warped_duration = tone_data[tone_pc].duration * frame_msec / 1000UL;
      tone( TONE_PIN, tone_data[tone_pc].frequency, warped_duration );
      coDelay( warped_duration * 1.1 )
    }
  coEnd
  tone_data = NULL; // automatically end melody
}

void tonePlay( const melodyData *melody ) // start a melody
{
  tone_data = melody;
}

//----------------------------------------------------------------
// beeper

uint16_t beep_msec;

#define BEEPER_PIN 12

void beepTask() // run the beeper for the programmed time
{
  coBegin
    coWaitWhile(!digitalRead( BEEPER_PIN ))
    coDelay(beep_msec)
    digitalWrite( BEEPER_PIN, LOW );
  coEnd
}

void beep( uint16_t msec ) // start beeping
{
  digitalWrite( BEEPER_PIN, HIGH );
  beep_msec = msec;
}

//----------------------------------------------------------------
// speed knob

#define SPEED_POLL_MSEC 100 // 10/sec

void speedTask() // read the speed knob, convert to frames/sec
{
  coBegin
    frame_msec = map( analogRead(POT_PIN), 0,1023, 2000,500 );
    coDelay(SPEED_POLL_MSEC)
  coEnd
}

//----------------------------------------------------------------
// joystick with press

#define JOYSTICK_POLL_MSEC 50 // 20/sec

uint16_t joystick_x;
uint16_t joystick_y;

void joystickTask() // read joystick pots
{
  coBegin
    joystick_x = analogRead(JOY_X_PIN);
    joystick_y = analogRead(JOY_Y_PIN);
    coDelay(JOYSTICK_POLL_MSEC)
  coEnd
}

#define JOY_PIN 39

boolean  joystick_press = false;
uint32_t joystick_tm;

void joystickPressTask() // debounce joystick press
{
  coBegin
    if (!digitalRead( JOY_PIN ))
    { joystick_press = true;
      coDebounce( digitalRead( JOY_PIN ), 50 )
    }
  coEnd
}

//----------------------------------------------------------------
// DHT11 temp/humidity sensor

#define DHT_POLL_MSEC 10000 // DHT11 data refresh rate

#include "DHT_Async.h" // https://github.com/toannv17/DHT-Sensors-Non-Blocking
#define DHT_PIN 4
#define DHT_SENSOR_TYPE DHT_TYPE_11
DHT_Async dht_sensor(DHT_PIN, DHT_SENSOR_TYPE);

float dht_F;
float dht_RH;

void dhtTask() // poll the DHT11 sensor
{
  coBegin
    dht_sensor.measure( & dht_F, & dht_RH );
    coDelay(DHT_POLL_MSEC)
  coEnd
}

//----------------------------------------------------------------
// NTC temperature sensor

#define NTC_POLL_MSEC 2000

float ntc_F = 0;

void ntcTask() // read the NTC, convert to C
{
  uint16_t adc;
  double tempK;
  float  tempC;
  coBegin
    adc = analogRead( NTC_PIN );
    tempK = log(10000.0 * ((1024.0 / adc - 1)));
    tempK = 1 / (0.001129148 + (0.000234125 + (0.0000000876741 * tempK * tempK )) * tempK ); //  Temp Kelvin
    tempC = tempK - 273.15;      // Convert Kelvin to Celcius
    ntc_F = (tempC * 9.0)/ 5.0 + 32.0; // Convert Celcius to Fahrenheit
    coDelay(NTC_POLL_MSEC)
  coEnd
}

//----------------------------------------------------------------
// rain sensor

#define RAIN_POLL_MSEC 1000

uint16_t rain_amount = 0;

void rainTask() // read the water sensor
{
  coBegin
    rain_amount = analogRead(RAIN_PIN);
    coDelay(RAIN_POLL_MSEC)
  coEnd
}

//----------------------------------------------------------------
// light sensor

#define LDR_POLL_MSEC 100

uint16_t ldr1_amount = 0;
uint16_t ldr2_amount = 0;

void ldrTask() // read both LDR's
{
  coBegin
    ldr1_amount = analogRead(LDR1_PIN);
    ldr2_amount = analogRead(LDR2_PIN);
    coDelay(LDR_POLL_MSEC)
  coEnd
}

//----------------------------------------------------------------
// sound sensor

#define SOUND_POLL_MSEC 50 // 20/sec

uint16_t sound_amount = 0;

void soundTask() // read the sound sensor
{
  coBegin
    sound_amount = analogRead(SOUND_PIN);
    srDisplay(map( sound_amount, 0,1023, 0,7 )); // 8 LEDs will represent distance
    cpDisplay(map( sound_amount, 0,1023, 0,15)); // 15 LEDs will represent distance
    coDelay(SOUND_POLL_MSEC)
  coEnd
}

//----------------------------------------------------------------
// '595 shift register to individual LEDs

#define LED_DOT        1
#define LED_BAR        2
#define LED_FLASH      3
#define LED_CHASER_DOT 4
#define LED_CHASER_BAR 5
#define LED_CYLON      6

#define RCLK_PIN 10 // plus SPI SCK,MOSI

byte     sr_sketch = 0; // active sketch
int8_t   sr_index;      // data for sketch
byte     sr_data;
uint16_t sr_rate;       // flash or chaser rate
CoroutineContext srCC = {0,0};

void srWrite( byte value )
{
  SPI.beginTransaction( SPISettings(10000000, MSBFIRST, SPI_MODE0) );
    digitalWrite( RCLK_PIN, LOW );
      SPI.transfer( value );
    digitalWrite( RCLK_PIN, HIGH );
  SPI.endTransaction();
}

void srAnimationTask()
{
  coBeginX(srCC)
    if (sr_sketch == LED_DOT)
    { // constant (dot)
      byte d = 0;
      if (sr_index) d = 1 << (sr_index-1);
      srWrite( d );
    }
    else if (sr_sketch == LED_BAR)
    { // constant (bar)
      byte d = 0;
      for (byte i=0; i < sr_index; i++) d |= (1 << i); 
      srWrite( d );
    }
    else if (sr_sketch == LED_FLASH)
    { // flash
      for (;true;)
      { srWrite( 0xFF );
        coDelayWarpedX(srCC, sr_rate);
        srWrite( 0 );
        coDelayWarpedX(srCC, sr_rate);
      }
    }
    else if (sr_sketch == LED_CHASER_DOT)
    { // chaser (dot)
      for (sr_index=0; true; sr_index++)
      { srWrite( 1 << (sr_index & 7) );
        coDelayWarpedX(srCC, sr_rate);
      }
    }
    else if (sr_sketch == LED_CHASER_BAR)
    { // chaser (bar)
      sr_data = 0; // chaser bitmap
      for (sr_index=0; true; sr_index = (sr_index+1) % 8)
      { sr_data ^= 1 << sr_index;
        srWrite( sr_data );
        coDelayWarpedX(srCC, sr_rate);
      }
    }
    else if (sr_sketch == LED_CYLON)
    { // cylon (reversing chaser)
      for (;true;)
      { for (sr_index=0; sr_index < 8; sr_index++)
        { srWrite( 1 << sr_index );
          coDelayWarpedX(srCC, sr_rate);
        }
        for (sr_index=7; sr_index >= 0; sr_index--)
        { srWrite( 1 << sr_index );
          coDelayWarpedX(srCC, sr_rate);
        }
      }
    }
    sr_sketch = 0; // automatically end animation
  coEndX(srCC)
}

void srAnimationStart( byte sketch, uint16_t rate_msec )
{
  sr_sketch = sketch;
  sr_rate   = rate_msec;
  coResetX(srCC) // reset coroutine
}

void srDisplay( byte value )
{
  sr_sketch = LED_BAR;
  sr_index  = value;
  coResetX(srCC) // reset coroutine
} 

//----------------------------------------------------------------
// charliplexed LEDs
// handler code is in the 1msec interrupt

volatile uint16_t cp_buffer = 0; // 15 bit LED output buffer

byte     cp_sketch = 0; // active sketch
int8_t   cp_index;      // data for sketch
uint16_t cp_data;
uint16_t cp_rate;       // flash or chaser rate
CoroutineContext cpCC = {0,0};

void cpAnimationTask()
{
  coBeginX(cpCC)
    if (cp_sketch == LED_DOT)
    { // constant (dot)
      cp_buffer = 1 << cp_index;
    }
    else if (cp_sketch == LED_BAR)
    { // constant (bar)
      byte d = 0;
      for (byte i=0; i < cp_index; i++) d |= 1 << cp_index;
      cp_buffer = d;
    }
    else if (cp_sketch == LED_FLASH)
    { // flash a few times
      for (cp_index=0; cp_index < 5; cp_index++)
      { cp_buffer = 0xFF;
        coDelayWarpedX(cpCC, cp_rate);
        cp_buffer = 0;
        coDelayWarpedX(cpCC, cp_rate);
      }
    }
    else if (cp_sketch == LED_CHASER_DOT)
    { // chaser (dot)
      for (cp_index=0; true; cp_index = (cp_index+1) % 15)
      { cp_buffer = (uint16_t)1 << cp_index;
        coDelayWarpedX(cpCC, cp_rate);
      }
    }
    else if (cp_sketch == LED_CHASER_BAR)
    { // chaser (bar)
      cp_data = 0; // chaser bitmap
      for (cp_index=0; true; cp_index = (cp_index+1) % 15)
      { cp_data ^= 1 << cp_index;
        cp_buffer = cp_data;
        coDelayWarpedX(cpCC, cp_rate);
      }
    }
    else if (cp_sketch == LED_CYLON)
    { // cylon (reversing chaser)
      for (;true;)
      { for (cp_index=0; cp_index < 16; cp_index++)
        { cp_buffer = (uint16_t)1 << sr_index;
          coDelayWarpedX(cpCC, cp_rate);
        }
        for (cp_index=15; cp_index >= 0; cp_index--)
        { cp_buffer = (uint16_t)1 << sr_index;
          coDelayWarpedX(cpCC, cp_rate);
        }
      }
    }
    cp_sketch = 0; // automatically end animation
  coEndX(cpCC)
}

void cpAnimationStart( byte sketch, uint16_t rate_msec )
{
  cp_sketch = sketch;
  cp_rate   = rate_msec;
  coResetX(cpCC) // reset coroutine
}

void cpDisplay( byte value )
{
  cp_sketch = LED_BAR;
  cp_index  = value;
  coResetX(cpCC) // reset coroutine
} 

//----------------------------------------------------------------
// NOT DS3231 RTC
// DS1307 RTC

#include "RTClib.h" // https://github.com/adafruit/RTClib
RTC_DS1307 rtc;

DateTime rtc_now;

void rtcTask() // read the RTC
{
  coBegin
      rtc_now = rtc.now();
      coDelay(1000)
  coEnd
}

//----------------------------------------------------------------
// IMU

#include "MPU6050.h" // from Elegoo tutorial library
MPU6050 mpu;

#define IMU_POLL_MSEC 100

Vector imu_accel;

void imuTask() // read the MPU6050 accelerometer
{
  coBegin
    imu_accel = mpu.readRawAccel();
    coDelay(IMU_POLL_MSEC)
  coEnd
}

//----------------------------------------------------------------
// IR remote control
// handler code is in library using timer interrupt

#define IR_PIN 41
#define DECODE_NEC      // Includes Apple and Onkyo. To enable all protocols , just comment/disable this line.
#include <IRremote.hpp> // https://github.com/Arduino-IRremote/Arduino-IRremote

//----------------------------------------------------------------
// 16x2 LCD

#include <LiquidCrystal.h>
#define LCD_RS     16
#define LCD_ENABLE 17
#define LCD_D4      6
#define LCD_D5      7
#define LCD_D6      8
#define LCD_D7      9
LiquidCrystal lcd( LCD_RS, LCD_ENABLE, LCD_D4, LCD_D5, LCD_D6, LCD_D7 );

#define LCD_POPUP_MSEC 3000

char     lcd_buffer[2*16];
char     lcd_live  [2*16];
boolean  lcd_preempt = false;

void lcdTask() // update the hardware LCD from the active buffer
{
  int8_t cursorat = -1;
  coBegin
    if (lcd_preempt)
      coDelay(LCD_POPUP_MSEC)
    // update live display from buffer
    for (byte i=0; i < 2*16; i++)
      if (lcd_live[i] != lcd_buffer[i])
      { if (i != cursorat) lcd.setCursor( i%16, i/16 );
        lcd.write( lcd_live[i] = lcd_buffer[i] );
        cursorat = i+1;
      }
    coDelay(50); // limit frame update rate
  coEnd
}

void lcdPopup( const char *text1, const char *text2 ) // temporary message
{
  lcd.clear();
  lcd.print(text1);
  lcd.setCursor(0,1);
  lcd.print(text2); 
  lcd_preempt = true;
}

void lcd_formatInt( byte row, byte col, int value, byte width )
{
  byte addr = row*16 + col;
  char buffer[10]; // Buffer to hold formatted string
  dtostrf( value, 1, 0, buffer );
  if (strlen(buffer) > width)
    for (byte i=0; i < width; i++) lcd_buffer[addr+i] = '*'; // overflow
  else
  { byte x;
    for (x=0; x < width-strlen(buffer); x++) lcd_buffer[addr+x] = ' ';  
    for (byte i=0; i < strlen(buffer); i++) lcd_buffer[addr+x+i] = buffer[i];
  }
}

void lcd_formatFloat( byte row, byte col, float value )
{
  // Format the float value within 4 columns
  char buffer[10]; // Buffer to hold formatted string
  dtostrf( value, 1, 1, buffer );
  // stuff into LCD buffer
  byte addr = row*16 + col;
  byte i;
  for (i=0; i < 4 && buffer[i]; i++) lcd_buffer[addr+i] = buffer[i];
  for (; i < 4; i++) lcd_buffer[addr+i] = ' ';
}

//----------------------------------------------------------------
// ultrasonic distance sensor

#define ULTRASONIC_POLL_MSEC 100 // refresh rate

#define TRIG_PIN 15
#define ECHO_PIN 14

uint16_t distance_cm = 0;

unsigned long ultrasonicMeasure() // take a measurement
{
  // The PING))) is triggered by a HIGH pulse of 2 or more microseconds.
  digitalWrite(TRIG_PIN, HIGH);
  delayMicroseconds(10);
  digitalWrite(TRIG_PIN, LOW);
  long duration = pulseIn(ECHO_PIN, HIGH);
  return (duration*.0343)/2; // convert to cm
}

void ultrasonicTask() // sample the ultrasonic distance sensor
{
  coBegin
    distance_cm = ultrasonicMeasure();
    coDelay(ULTRASONIC_POLL_MSEC)
    cpDisplay( map(distance_cm, 0,100, 0,15) ); // 15 LEDs will represent distance
  coEnd
}
  
//----------------------------------------------------------------
// servo controller

#include <PWMServo.h> // https://github.com/PaulStoffregen/PWMServo
#define SERVO_PIN 3
PWMServo myservo;  // create servo object to control a servo

byte servo_control = 0; // b7=enable, b6=command vs. position, b5..b0=data

void servoTask() // run servo commands
{
  if ((servo_control & 0x80) == 0) return;
  if ((servo_control & 0x40) == 0)
  { myservo.write( map( servo_control & 0x3F, 0,63, 0,180 ) );
    servo_control = 0;
  }
  else // command
  { static int i;
    coBegin
      if ((servo_control & 0x3F) == 0)
      { // rise and wave
        for (i=0; i < 5; i++)
        { myservo.write( 90+20 );
          coDelayWarped(500)
          myservo.write( 90-20 );
          coDelayWarped(500)
        }
        myservo.write( 0 );
      }
      else if ((servo_control & 0x3F) == 1)
      { // open & close gate
        myservo.write( 90 );
        coDelayWarped(5000)
        myservo.write( 0 );
      }
      servo_control = 0;
    coEnd
  } 
}

//----------------------------------------------------------------
// fan motor controller (L298)

#define MOT_1 44 // PL5, PWM capable
#define MOT_2 43 // PL6

byte fan_control = 0; // b7=enabled, b6=direction, b5..b0=speed

void fanTask() // run fan commands
{
  if (fan_control & 0x80) // enable
  {
    if (fan_control & 0x40) // direction
    { digitalWrite( MOT_1, 0 );
      digitalWrite( MOT_2, 1 );
    }
    else // forward, speed controlled
    { analogWrite( MOT_1, map( fan_control & 0x3F, 0,63, 0,255 ) );
      digitalWrite( MOT_2, 1 );
    }
  }
  else // disabled
  { digitalWrite( MOT_1, 0 );
    digitalWrite( MOT_2, 0 );
  }
}

//----------------------------------------------------------------
// stepper motor controller (UNL2003)

const byte stepper_phase[8] = { 0x01, 0x03, 0x02, 0x06, 0x04, 0x0C, 0x08, 0x09 };
byte stepper_control = 0; // b7=enable, b6=direction, b5..0=speed
byte stepper_index   = 0;

void stepperTask() // run the stepper motor
{
  coBegin
    if (stepper_control & 0x80) // enabled ?
      do
      { PORTK = (PORTK & 0x0F) | (stepper_phase[stepper_index] << 4);
        stepper_index = (stepper_index+((stepper_control & 0x40) ? 7:1)) % 8;
        coDelayWarped( (uint32_t)(map( stepper_control & 0x3F, 0,63, 500,10 )) )
      } while (stepper_control & 0x80);
    else
      PORTK &= 0x0F; // turn off all stepper outputs
  coEnd
}

//----------------------------------------------------------------
// relay controller

#define RELAY_ON     1
#define RELAY_OFF    2
#define RELAY_TOGGLE 3

#define MAX_RELAY_MSEC 50 // max relay switching rate

#define  RELAY_PIN 42
byte     relay_control = 0; // high 3 bits: 1=off, 2=on, 3=toggling (low 6 bits are speed)
uint32_t relay_tm;
CoroutineContext relayCC;

void relayTask() // run relay commands
{
  static byte relay_live = 0;
  coBeginX(relayCC)
    coWaitWhileX(relayCC, relay_control == relay_live )
    // new relay command
    relay_live = relay_control;
    if ((relay_control >> 6) == RELAY_TOGGLE)
      for (;true;)
      { digitalWrite( RELAY_PIN, !digitalRead(RELAY_PIN) );
        coDelayWarpedX(relayCC, (relay_control & 0x3F) * 100UL)
      }
    if      ((relay_control >> 6) == RELAY_ON)  digitalWrite( RELAY_PIN, LOW );
    else if ((relay_control >> 6) == RELAY_OFF) digitalWrite( RELAY_PIN, HIGH );
    // limit max relay switching rate
    coDelayWarpedX(relayCC, MAX_RELAY_MSEC )
  coEndX(relayCC)
}

void relayCommand( byte command, uint16_t msecX100 = 0 ) // max 6.3 sec toggle rate
{
  if (msecX100 > 63) return;
  relay_control = (command << 6) + msecX100;
  coResetX(relayCC) // reset coroutine
}

//----------------------------------------------------------------
// PIR motion sensor

#define PIR_PIN 40

void PIRTask()
{
  coBegin
    coWaitWhile(digitalRead( PIR_PIN ));
    // do animation sequence
    servo_control = 0xC0; // enable, command 0 = rise & wave
    tonePlay( test_melody );
    // wait for PIR inactive for 2 minutes
    coDebounce( digitalRead( PIR_PIN ), 2*60*1000UL )
  coEnd
}

//----------------------------------------------------------------
// buttons on resistor ladder

struct
{ boolean pressed;
  byte    index;
} button = { false,0 };

void buttonTask() // sets button.pressed
{
  uint16_t adc;
  int8_t   i;
  coBegin
    adc = analogRead(BTN_PIN);
    i   = -1;
    if      (adc <  85) i = 0; // identify button
    else if (adc < 256) i = 1;
    else if (adc < 426) i = 2;
    else if (adc < 682) i = 3;
    else if (adc < 938) i = 4;
    if (i >= 0)
    { button.index   = i;
      button.pressed = true;
      coDebounce( analogRead(BTN_PIN) > 937, 50 ); // wait for button release
    }
  coEnd
}

//----------------------------------------------------------------
// rotary encoder & press
// handler code is in the 1msec interrupt

#include "EncoderSW.h"
EncoderSW knob;

//----------------------------------------------------------------
// 4x4 keypad
// NOT handler code is in the 1msec interrupt

struct
{ boolean pressed;
  byte    row;
  byte    col;
  boolean state;
  uint32_t tm;
} keypress = { false,0,0,false,0 };

void keypadScanTask()
{  
  // scan the keypad
  { static byte scan_index = 0;
    byte inp = PORTA; // column in low nybble
    // determine which col is hit
    byte c; for (c=0; c < 4; c++)
      if ((inp & (1 << c)) == 0)
      { if (!keypress.state)
        { // key press event
          keypress.pressed = true;
          keypress.state = true;
          keypress.row = scan_index;
          keypress.col = c;
        }
        keypress.tm = millis();
      }
    if (c >= 4) // no keys pressed
      if (keypress.state) // release after debounce
        if (millis() - keypress.tm > 50) keypress.state = false;      
    scan_index = (scan_index+1) % 4;
    // select next row
    PORTA |= 0xF0; 
    PORTA ^= 0x80 >> scan_index;
  }
}

//----------------------------------------------------------------
// 7 segment LED displays, driven directly from ATmega

#define CC1_PIN 63
#define CC2_PIN 64
#define CC3_PIN 65
#define CC4_PIN 30
#define CC5_PIN 13

byte led_buffer[5]; // 4 digit + 1 digit
//    -A-
//  F/   /B
//   -G-
// E/   /C
//  -D-
byte SEG_TABLE[11] =
{ //  gfedcba
   0b00111111, // 0
   0b00000110, // 1
   0b01011011, // 2
   0b01001111, // 3
   0b01100110, // 4
   0b01101101, // 5
   0b01111100, // 6
   0b00000111, // 7
   0b01111111, // 8
   0b01100111, // 9
   0b00000000  // blank
};

//----------------------------------------------------------------
// 8x8 LED array, driven by MAX7219

#define LOAD_PIN 2 // MAX7219 /CS

#define MATRIX_UPDATE_MSEC 20 // limit max update rate

byte matrix_buffer[8];
byte matrix_live  [8];

void WriteMax7219(unsigned char address,unsigned char dat)
{
  SPI.beginTransaction( SPISettings(14000000, MSBFIRST, SPI_MODE0) );
    digitalWrite(LOAD_PIN,LOW);
      byte buf[2]; buf[0] = address; buf[1] = dat;
      SPI.transfer(buf,2);
    digitalWrite(LOAD_PIN,HIGH);
  SPI.endTransaction();
}

void WriteMax7219Buffer()
{
  SPI.beginTransaction( SPISettings(10000000, MSBFIRST, SPI_MODE0) );
    for (byte i=0; i < 8; i++)
    { 
      digitalWrite(LOAD_PIN,LOW);
        byte buf[2]; buf[0] = i+1; buf[1] = matrix_buffer[i];
        SPI.transfer(buf,2);
      digitalWrite(LOAD_PIN,HIGH);
    }
  SPI.endTransaction();
}

void matrixTask() // update LED array hardware from matrix_buffer
{
  coBegin
    for (byte i=0; i < 8; i++)
      if (matrix_buffer[i] != matrix_live[i])
        WriteMax7219( i+1, matrix_live[i] = matrix_buffer[i] );
    coDelay(MATRIX_UPDATE_MSEC)
  coEnd 
}

//----------------------------------------------------------------
// 1 msec interrupt

void TimerHandler()
{ 
  // multiplex 5 digit LED display
  // 1 msec / digit => 5 msec => 200 frames/sec
  { volatile uint8_t *cc_port[5] = { &PORTK, &PORTK, &PORTK, &PORTC, &PORTB };
    const byte        cc_bit [5] = {  1,      2,      3,      7,      7     };
    static byte mux_index = 0;   
    byte segs = SEG_TABLE[led_buffer[mux_index] & 0x0F];
    *cc_port[(mux_index+4)%5] &= ~(1 << cc_bit[(mux_index+4)%5]); // turn off previous digit
    PORTC = segs;
    *cc_port[mux_index]       |=   1 << cc_bit[mux_index]; // turn on this digit
    mux_index = (mux_index+1) % 5;
  }

  // charlieplex 15 LEDs
  // 1 msec / LED => 15 msec => 66 frames/sec
  { static byte cp_index = 0;
    const byte cp_dat[12] = { 0b0001,0b0010,0b0100, 0b0001,0b0010,0b1000, 0b0001,0b0100,0b1000, 0b0010,0b0100,0b1000 };  
    const byte cp_ddr[12] = { 0b1001,0b1010,0b1100, 0b0101,0b0110,0b1100, 0b0011,0b0110,0b1010, 0b0011,0b0101,0b1001 };
    if (cp_buffer & (1 << cp_index))
    { PORTL = (PORTL & 0xF0);
      DDRL  = (DDRL  & 0xF0) | cp_ddr[cp_index];
      PORTL |= cp_dat[cp_index];
    }
    else // off
    { PORTL = (PORTL & 0xF0);
      DDRL  = (DDRL  & 0xF0);
    }
    cp_index = (cp_index+1) % 12;
  } 
}

//----------------------------------------------------------------
// 7seg display manager

#define POPUP_7SEG_MSEC 1000

byte    led7x1_value = 0;
boolean led7x1_popup = false;

void led7x1_Popup( byte value ) // pop up value for a few seconds
{
  led_buffer[4] = value;
  led7x1_popup = true;
}
void led7x1_Display( byte value ) // normal display value
{
  led_buffer[4] = value;
  led7x1_value = value;  
  led7x1_popup = false;
}
void led7x1_Task() // run the 7seg LED manager
{
  coBegin
    if (led7x1_popup)
    { coDelay(POPUP_7SEG_MSEC)
      led_buffer[4] = led7x1_value;
      led7x1_popup  = false;
    }
  coEnd  
}

int     led7x4_value = 0;
boolean led7x4_popup = false;

void led7x4_Format(int value)
{  
  String s = String(value);
  if (s.length() > 4) // show "9999" for overflow
    led_buffer[0] = led_buffer[1] = led_buffer[2] = led_buffer[3] = 9;
  else // 4 digit, leading zero suppressed
  { byte i=0;
    for (byte n=4-s.length(); n > 0; n--) led_buffer[i++] = 10; // blank
    for (byte n=0; n < s.length(); n++) led_buffer[i++] = s.charAt(n) - '0';
  }
}
void led7x4_Popup( int value ) // pop up value for a few seconds
{
  led7x4_Format(value);
  led7x4_popup = true;
}
void led7x4_Display( int value ) // normal display value
{
  led7x4_Format(value);
  led7x4_value = value;  
  led7x4_popup = false;
}
void led7x4_Task() // run the 7seg LED manager
{
  coBegin
    if (led7x4_popup)
    { coDelay(POPUP_7SEG_MSEC)
      led_buffer[4] = led7x4_value;
      led7x4_popup  = false;
    }
  coEnd  
}

//----------------------------------------------------------------
// Application

const char *mode_desc[] = { "CLOCK", "ENVIROMNENT", "SOUND", "DISTANCE", "IMU" };
#define MODE_CLOCK 0
#define MODE_ENV   1
#define MODE_SOUND 2
#define MODE_DIST  3
#define MODE_IMU   4
byte mode = MODE_CLOCK;

//----------------------------------------------------------------
//-- Main --------------------------------------------------------
//----------------------------------------------------------------

void setup() 
{
Serial.begin(115200);
Serial.println(F("Mega Fidget Started!"));

  // port A
  DDRA = 0xF0; // 4x4 keypad; outputs on high nybble
  PORTA= 0x7F; // 4x4 input pullups, enable row 0
  // port C
  DDRC = 0xFF; // 7seg LED segment outputs
  // port K
  DDRK = 0xFF; // stepper outputs, CP_LED outputs
  PORTK= 0x00; //   all off          all off
  // port L
  DDRL = 0xF0; // cp leds + motor directions, + relay

  // ultrasonic sensor
  pinMode( TRIG_PIN, OUTPUT ); 
  pinMode( ECHO_PIN, INPUT );

  // MAX7219 LED aray
  pinMode( LOAD_PIN, OUTPUT ); // MAX7219

  // '595 shift register
  pinMode( RCLK_PIN, OUTPUT ); // '595 SR

  // PIR sensor
  pinMode( PIR_PIN, INPUT );

  // joystick press
  pinMode( JOY_PIN, INPUT_PULLUP );

  // IR receiver
  pinMode( IR_PIN,  INPUT_PULLUP );

  // 7 segment LED cathodes
  pinMode( CC1_PIN, OUTPUT ); digitalWrite( CC1_PIN, LOW );
  pinMode( CC2_PIN, OUTPUT ); digitalWrite( CC2_PIN, LOW );
  pinMode( CC3_PIN, OUTPUT ); digitalWrite( CC3_PIN, LOW );
  pinMode( CC4_PIN, OUTPUT ); digitalWrite( CC4_PIN, LOW );
  pinMode( CC5_PIN, OUTPUT ); digitalWrite( CC5_PIN, HIGH ); // POST LED on

#define POST(n) PORTC = SEG_TABLE[n]; // show power up step on 7seg LED

  POST(0) // show power up step on 7seg LED

  // LCD
  lcd.begin(16,2);
  for (byte i=0; i < 2*16; i++) { lcd_buffer[i] = ' ';  lcd_live[i] = 0; }

  POST(1) // show power up step on 7seg LED

  // rotary encoder
  knob.begin(); // rotary encoder with press

  POST(2) // show power up step on 7seg LED

  // servo
  myservo.attach(SERVO_PIN);

  POST(3) // show power up step on 7seg LED

  // RTC
  Wire.begin(); // RTC, IMU

  for (byte i=0; i < 10; i++)
    if (!rtc.begin()) { Serial.println(F("Can't find RTC")); delay(100); }
    else { rtc.adjust(DateTime(F(__DATE__), F(__TIME__))); break; }

  POST(4) // show power up step on 7seg LED

  SPI.begin();  // '595, MAX7219, MFRC522

  // '595 shift register
  srWrite(0x00); // all LEDs off

  // LED array
  WriteMax7219(0x09, 0x00);       //decoding ：none
  WriteMax7219(0x0a, 0x03);       //brightness 
  WriteMax7219(0x0b, 0x07);       //scanlimit；8 LEDs
  WriteMax7219(0x0c, 0x01);       //power-down mode：0，normal mode：1
  WriteMax7219(0x0f, 0x00);       //test display：1；EOT，display：0
  for (byte i=0; i < 8; i++) matrix_buffer[i] = matrix_live[i] = 0;
  WriteMax7219Buffer();

  POST(5) // show power up step on 7seg LED

  // IR receiver
  IrReceiver.begin(IR_PIN);

  POST(6) // show power up step on 7seg LED

  // MPU6500 IMU (it's what came with the kit, not a MPU6050)
  if (!mpu.begin(MPU6050_SCALE_2000DPS, MPU6050_RANGE_2G)) { Serial.println(F("Can't find MPU6050")); while(1); }

  POST(7) // show power up step on 7seg LED

  // RFID reader
  mfrc522.PCD_Init();  // Init MFRC522 card

  POST(8) // show power up step on 7seg LED

  // blank 7seg & charliplex LED buffers
  for (byte i=0; i < 5; i++) led_buffer[i]=10; // blanks
  cp_buffer = 0;

  // 1 msec timer for LED refresh
  ITimer1.init();
  if (!ITimer1.attachInterruptInterval(TIMER_INTERVAL_MS, TimerHandler)) { Serial.println(F("Can't set ITimer1")); while (1); }

  POST(9) // show power up step on 7seg LED
}

//----------------------------------------------------------------

void shutdownAll()
{
  // LED array
  for (byte i=0; i < 8; i++) matrix_buffer[i] = matrix_live[i] = 0;
  WriteMax7219Buffer();
  // '595 shift register
  srWrite(0x00);
  // blank 7seg & charliplex LED buffers
  for (byte i=0; i < 5; i++) led_buffer[i]=10; // blanks
  cp_buffer = 0;
  // relay off
  digitalWrite( RELAY_PIN, LOW );
  // fan off
  digitalWrite( MOT_1, 0 );
  digitalWrite( MOT_2, 0 );
  // LCD
  lcd.clear();
}

void loop() 
{
  static boolean diagnostic_mode = false;
  // toggle diagnostic mode when '/' is entered  
  int c = Serial.read();  
  if (c == '/') 
  { shutdownAll();
    diagnostic_mode = !diagnostic_mode;
    Serial.println( diagnostic_mode ? F("Diagnostic Mode") : F("Normal Mode") );
    if (diagnostic_mode) lcd.print("DIAGNOSTIC MODE");
  }
  if (diagnostic_mode) serialConsole(c); else normalMode();
}

//----------------------------------------------------------------
//-- Diagnotic Console -------------------------------------------
//----------------------------------------------------------------

const struct // diagnostic commands
{ char   key;
  String desc;
} commands[] = { {'5',"'595 SR"},           // 0
                 {'7',"7 Segment Display"}, // 1
                 {'a',"Rain Sensor"},       // 2
                 {'b',"Beeper"},            // 3
                 {'c',"Charlieplex"},       // 4
                 {'d',"DHT11 Sensor"},      // 5
                 {'f',"Fan"},               // 6
                 {'i',"I2C Bus Scan"},      // 7
                 {'j',"Joystick"},          // 8
                 {'l',"LDR Sensors"},       // 9
                 {'m',"Max7219 LED Array"}, // 10
                 {'n',"NTC"},               // 11
                 {'o',"Microphone"},        // 12
                 {'p',"PIR Sensor"},        // 13
                 {'k',"Speaker"},           // 14
                 {'r',"Relay"},             // 15
                 {'s',"Speed Knob"},        // 16
                 {'t',"Time"},              // 17
                 {'u',"IMU"},               // 18
                 {'v',"Servo"},             // 19
                 {'z',"Stepper"} };         // 20
                  // auto tests (continuously monitored):
                  //   rotary encoder
                  //   keypad
                  //   buttons
                  //   IR remote
                  //   RFID
#define NBR_COMMANDS 21
#define CMD_595SR     0
#define CMD_7SEG      1
#define CMD_RAIN      2
#define CMD_BEEP      3
#define CMD_CP        4
#define CMD_DHT       5
#define CMD_FAN       6
#define CMD_I2C       7
#define CMD_JOY       8
#define CMD_LDR       9
#define CMD_MAX7219  10
#define CMD_NTC      11
#define CMD_SOUND    12
#define CMD_PIR      13
#define CMD_SPEAKER  14
#define CMD_RELAY    15                   
#define CMD_SPEED    16
#define CMD_TIME     17
#define CMD_IMU      18
#define CMD_SERVO    19
#define CMD_STEPPER  20

boolean active_test[NBR_COMMANDS]; // task enables

int16_t rotary_number = 0; // rotary encoder spin value

void CPtestTask() // charliplex LED test
{
  static byte i;
  coBegin
    for (i=0; i < 12; i++)
    { cp_buffer = 1 << i;
      coDelay(250)
    }
  coEnd
}
void SegTestTask() // 7 segment LED test
{
  static uint16_t i;
  coBegin
    for (i=0; i < 10000; i++)
    { led7x4_Display(i);
      led7x1_Display(i % 10);
      coDelay(100)
    }
  coEnd
}
void MaxTestTask() // MAX7219 LED array test
{
  static byte x,y;
  coBegin
    for (y=0; y < 8; y++)
      for  (x=0; x < 8; x++)
      { matrix_buffer[y] = 1 << x;
        WriteMax7219Buffer();
        coDelay(100)
      }
  coEnd
}
void SRtestTask() // '595 shift register LED test
{
  static byte i;
  coBegin
    for (i=0; i < 8; i++)
    { srWrite(1 << i);
      coDelay(500)
    }
  coEnd
}
void RelayTestTask() // relay test
{
  coBegin
    digitalWrite( RELAY_PIN, HIGH );
    coDelay( 1000 );
    digitalWrite( RELAY_PIN, LOW );
    coDelay( 1000 )
  coEnd
}
void FanTestTask() // fan test
{
  static uint16_t i;
  coBegin
    digitalWrite( MOT_2, 0 );
    for (i=0; i < 256; i++)
    { analogWrite( MOT_1, i );
      coDelay(100) // 26 second ramp up time
    }
    coDelay(1000)
    digitalWrite( MOT_1, 0 );
    digitalWrite( MOT_2, 1 );
    coDelay(1000)
  coEnd      
}
void PIRtestTask() // PIR sensor test
{
  digitalWrite( RELAY_PIN, digitalRead(PIR_PIN) );
}
void joystickTestTask() // joystick test
{
  coBegin
    Serial.print("JOY: "); 
    Serial.print(analogRead(JOY_X_PIN)); Serial.print(","); 
    Serial.print(analogRead(JOY_Y_PIN)); Serial.print(", ");
    Serial.println(digitalRead(JOY_PIN));
    coDelay(250)
  coEnd
}
void clockTestTask() // clock readout test
{
  coBegin
    { DateTime now = rtc.now();
      Serial.print(now.year(), DEC);
      Serial.print('/');
      Serial.print(now.month(), DEC);
      Serial.print('/');
      Serial.print(now.day(), DEC);
      Serial.print("  ");
      Serial.print(now.hour(), DEC);
      Serial.print(':');
      Serial.print(now.minute(), DEC);
      Serial.print(':');
      Serial.print(now.second(), DEC);
      Serial.println();
    }
    coDelay(1000)
  coEnd  
}
void IMUtestTask() // IMU readout test
{
  Vector imu_accel;
  coBegin
    imu_accel = mpu.readRawAccel();
    Serial.print("IMU: "); Serial.print(imu_accel.XAxis);
    Serial.print(", "); Serial.print(imu_accel.YAxis);
    Serial.print(", "); Serial.println(imu_accel.ZAxis);
    coDelay(500)
  coEnd
}
void buttonTestTask() // button detection
{
  static int8_t pressed_btn = -1;
  uint16_t adc = analogRead(BTN_PIN);
  int8_t i = -1;
  if      (adc <  85) i = 0; // identify button
  else if (adc < 256) i = 1;
  else if (adc < 426) i = 2;
  else if (adc < 682) i = 3;
  else if (adc < 938) i = 4;
  if (i >= 0)
    if (pressed_btn != i) { Serial.print("Button "); Serial.print(i+1); Serial.print(" ADC "); Serial.println(adc); }
  pressed_btn   = i;
}
void BeepTestTask()
{
  coBegin
    digitalWrite( BEEPER_PIN, HIGH );
    coDelay(250)
    digitalWrite( BEEPER_PIN, LOW );
    coDelay(1000)
  coEnd
}
void SpeakerTestTask()
{
  coBegin
    tone( TONE_PIN, 250, 250 );
    coDelay(250)
    tone( TONE_PIN, 500, 250 );
    coDelay(250)
    tone( TONE_PIN, 750, 250 );
    coDelay(250)
  coEnd  
}
void LDRtestTask()
{
  coBegin
    Serial.print("LDR: "); Serial.print(analogRead(LDR1_PIN)); Serial.print(", "); Serial.println(analogRead(LDR2_PIN));
    coDelay(1000)
  coEnd
}
void RainTestTask()
{
  coBegin
    Serial.print("Rain: "); Serial.println(analogRead(RAIN_PIN));
    coDelay(1000)
  coEnd
}
void SoundTestTask()
{
  coBegin
    Serial.print("Sound: "); Serial.println(analogRead(SOUND_PIN));
    coDelay(1000)
  coEnd
}
void NTCtestTask()
{
  coBegin
    Serial.print("NTC: "); Serial.println(analogRead(NTC_PIN));
    coDelay(1000)
  coEnd
}
void SpeedTestTask()
{
  coBegin
    Serial.print("Speed: "); Serial.println(analogRead(POT_PIN));
    coDelay(250)
  coEnd
}
void DHTtestTask()
{
  float dht_F  = 0;
  float dht_RH = 0;
  coBegin
    dht_sensor.measure( & dht_F, & dht_RH );
    Serial.print("DHT: "); Serial.print(dht_F); Serial.print("  "); Serial.println(dht_RH);
    coDelay(10000)
  coEnd
}
void servoTestTask()
{
  static int pos = 0;
  coBegin
    for (pos = 0; pos <= 180; pos += 1) 
    { myservo.write(pos); 
      coDelay(15)
    }
    for (pos = 180; pos >= 0; pos -= 1) 
    { myservo.write(pos); 
      coDelay(15)
    }
  coEnd
}
void stepperTestTask()
{
  static int stepper_index = 0;
  static byte i = 0;
  coBegin
    for (i=0; i < 100; i++)
    { PORTK = (PORTK & 0x0F) | (stepper_phase[stepper_index] << 4);
      stepper_index = (stepper_index+1) % 8;
      coDelay(50)    
    }
    for (i=0; i < 100; i++)
    { PORTK = (PORTK & 0x0F) | (stepper_phase[stepper_index] << 4);
      stepper_index = (stepper_index+7) % 8;
      coDelay(50)    
    }
  coEnd
}


void serialConsole( int c ) // Serial Monitor debug console
{
  for (byte cmd=0; cmd < NBR_COMMANDS; cmd++) if (commands[cmd].key == c)
  {
    active_test[cmd] = !active_test[cmd]; 
     
    switch (tolower(c))
    {
      //I2C device found at address 0x68 (clock)
      //I2C device found at address 0x69 (IMU)
      case 'i': // I2C bus scan
      { Serial.println(F("Scanning I2C Bus..."));
        for(byte address = 1; address < 127; address++ )
        { Wire.beginTransmission(address);
          if (Wire.endTransmission() == 0)
          { Serial.print(F("I2C device found at address 0x"));
            if (address<16) Serial.print("0");
            Serial.println(address,HEX);
          }
        }
        Serial.println(F("Done."));
        break;
      }
    case 'f': // fan test
      if (!active_test[cmd]) { digitalWrite( MOT_1, 0 ); digitalWrite( MOT_2, 0 ); }
      break;
    case 'r': // relay test
      if (!active_test[cmd]) digitalWrite( RELAY_PIN, LOW );
      if ( active_test[cmd]) active_test[CMD_PIR] = false; // override PIR test
      break;
    case '5': // SR test
      if (!active_test[cmd]) srWrite(0);
      break;
    case 'c': // CP LED test
      if (!active_test[cmd]) cp_buffer = 0;
      break;
    case 'm': // MAX7219 LED array test
      if (!active_test[cmd]) { for (byte y=0; y < 8; y++) matrix_buffer[y]=0; WriteMax7219Buffer(); }
      break;
    case '7':           // 7seg test
      if (!active_test[cmd]) for (byte i=0; i < 5; i++) led_buffer[i]=10; // blanks
      break;
    case 'p': // PIR test - triggers relay
      if (!active_test[cmd]) digitalWrite( RELAY_PIN, LOW );
      if ( active_test[cmd]) active_test[CMD_RELAY] = false; // override relay test
      break;
    case 'b': // beeper
      if (!active_test[cmd]) digitalWrite( BEEPER_PIN, LOW ); // turn off beeper
      break;
    case 'z': // stepper
      if (!active_test[cmd]) PORTK &= 0x0F; // turn off all stepper outputs
      break;
    case 'k': // speaker
    case 'j': // joystick test
    case 's': // speed knob test
    case 't': // clock test
    case 'x': // IMU test
      break;
    case '?':
      Serial.println(F("Mega Fidget Help"));
      Serial.println(F("'i' I2C Bus Scan"));
      for (byte i=0; i < NBR_COMMANDS; i++)
      { Serial.print("'"); Serial.print(commands[i].key); Serial.print("' "); Serial.print(commands[i].desc); 
        if (cmd != CMD_I2C) Serial.print(" Test");
        Serial.println();
      }
    }
    if (cmd != CMD_I2C) { Serial.print(commands[cmd].desc); Serial.println(active_test[cmd] ? F(" Test Started"):F(" Test Ended") ); }
  }

  // run selected diagnostic tasks
  if (active_test[CMD_CP])      CPtestTask();
  if (active_test[CMD_7SEG])    SegTestTask();
  if (active_test[CMD_MAX7219]) MaxTestTask();
  if (active_test[CMD_595SR])   SRtestTask();  // wow those fuckers are bright
  if (active_test[CMD_RELAY])   RelayTestTask();
  if (active_test[CMD_FAN])     FanTestTask();
  if (active_test[CMD_PIR])     PIRtestTask();
  if (active_test[CMD_JOY])     joystickTestTask();
  if (active_test[CMD_TIME])    clockTestTask();
  if (active_test[CMD_IMU])     IMUtestTask();
  if (active_test[CMD_BEEP])    BeepTestTask();
  if (active_test[CMD_SPEAKER]) SpeakerTestTask();
  if (active_test[CMD_SOUND])   SoundTestTask();
  if (active_test[CMD_RAIN])    RainTestTask();
  if (active_test[CMD_LDR])     LDRtestTask();
  if (active_test[CMD_NTC])     NTCtestTask();
  if (active_test[CMD_SPEED])   SpeedTestTask();
  if (active_test[CMD_DHT])     DHTtestTask();
  if (active_test[CMD_SERVO])   servoTestTask();
  if (active_test[CMD_STEPPER]) stepperTestTask();

  // run automatic monitoring tasks
  //   individual buttons
  buttonTestTask();
  //   rotary encoder with press
  knob.loop();
  int8_t spin = knob.getSpin(0);
  if (spin) { Serial.print(F("Spin: ")); Serial.println(spin); }
  if (knob.getPress(0)) Serial.println(F("Press"));
  //   keypad
  keypadScanTask();
  if (keypress.pressed) { Serial.print(F("Key: ")); Serial.print(keypress.col); Serial.print(","); Serial.println(keypress.row); keypress.pressed = false; }
  //   IR remote
  if (IrReceiver.decode())
  {
    if (IrReceiver.decodedIRData.protocol == UNKNOWN) {
        Serial.println(F("Received noise or an unknown (or not yet enabled) protocol"));
        // We have an unknown protocol here, print extended info
        IrReceiver.printIRResultRawFormatted(&Serial, true);
        IrReceiver.resume(); // Do it here, to preserve raw data for printing with printIRResultRawFormatted()
    } else {
        IrReceiver.resume(); // Early enable receiving of the next IR frame
        IrReceiver.printIRResultShort(&Serial);
        IrReceiver.printIRSendUsage(&Serial);
    }
    Serial.println();
  }
  //   RFID  
  if (mfrc522.PICC_IsNewCardPresent())
    if (mfrc522.PICC_ReadCardSerial())
    {
      Serial.print(F("Card UID:"));
      for (byte i = 0; i < mfrc522.uid.size; i++) 
      { Serial.print(mfrc522.uid.uidByte[i] < 0x10 ? " 0" : " ");
        Serial.print(mfrc522.uid.uidByte[i], HEX);
      } 
      Serial.println();
    }
}

//----------------------------------------------------------------
//-- Normal Operating Mode ---------------------------------------
//----------------------------------------------------------------

void normalMode()
{

//-- warped timekeeping

  { static uint32_t last_millis = millis();
    uint32_t tm = millis();
    uint32_t dt = tm - last_millis;
    my_millis += dt * 1000UL / frame_msec;
    last_millis = tm;
  }
  
//-- get inputs

  speedTask();        // => my_second
  joystickTask();     // => joystick_x, joystick_y
  joystickPressTask();// => joystick_press
  buttonTask();       // => button.pressed, button.index
  rtcTask();          // => rtc_now
  ntcTask();          // => ntc_amount
  rainTask();         // => rain_amount
  ldrTask();          // => ldr_amount
  dhtTask();          // => dht_F, dht_RH
  imuTask();          // => imu_accel
  knob.loop();        // rotary encoder with press
  keypadScanTask();   // 4x4 keypad
  
//-- process inputs

  soundTask();        // measure & display sound level
  ultrasonicTask();   // measure & display distance
  
  // mode select buttons
  if (button.pressed)
  { if (button.index != mode)
    { mode = button.index;
      // show mode on displays
      lcdPopup( mode_desc[mode], NULL );
      led7x1_Display( mode+1 );
      // beep each keypress
      beep(50);
      // start animations for mode
      switch (mode)
      { case MODE_CLOCK:
          relayCommand( RELAY_TOGGLE, 500 ); // toggle at 500 msec rate
          cpAnimationStart(LED_CHASER_DOT,1000); // 15 LEDs
          srAnimationStart(LED_CHASER_DOT,1000); // 8 LEDs
          break;
          
        case MODE_ENV:
          cpAnimationStart(LED_CHASER_BAR,200);
          srAnimationStart(LED_CHASER_BAR,200);
          break;
          
        case MODE_SOUND:
          srDisplay(0); // 8 LEDs will represent distance
          cpDisplay(0); // 15 LEDs will represent distance
          break;
          
        case MODE_DIST:
          srAnimationStart(LED_CYLON,1000); // 8 LEDs
          cpDisplay(0); // 15 LEDs will represent distance
          break;
          
        case MODE_IMU:
          cpAnimationStart(LED_CHASER_BAR,200);
          srAnimationStart(LED_CHASER_BAR,200);
          break;
      }
    }
    button.pressed = false;
  }

  // 4x4 keypad
  //  enters a number on the 7seg LEDs
  if (keypress.pressed)
  { static uint16_t keypad_number = 0;
    const byte KEYPAD_LUT[4][4] =
      { {'0','1','2','A'}, {'3','4','5','B'}, {'6','7','8','C'}, {'*','0','#','D' }};
    byte k = KEYPAD_LUT[keypress.row][keypress.col];
    keypress.pressed = false;
    // beep each keypress
    beep(50);
    switch (k)
    { case '#':
      case '*': keypad_number = 0; break;
      case 'A':
      case 'B':
      case 'C': // ........play a melody or something.......
      case 'D': break;
      default : keypad_number = keypad_number*10 + k-'0';
                if (keypad_number > 9999) keypad_number = 0;
    }
    if (mode == MODE_CLOCK || mode == MODE_DIST)
         led7x4_Popup(keypad_number);
    else led7x4_Display(keypad_number);
  }

  // RFID scanner
  if (mfrc522.PICC_IsNewCardPresent())
    if (mfrc522.PICC_ReadCardSerial())
    {
      beep(50);
      relayCommand( (mfrc522.uid.uidByte[0] & 1) ? RELAY_ON:RELAY_OFF );
      String s;
      for (byte i=0; i < mfrc522.uid.size; i++) s += String(mfrc522.uid.uidByte[i],HEX);
      lcdPopup("RFID",s.c_str());
    }
 
  // rotary encoder
  //  spins a number on the 7seg LEDs
  if (int8_t spin = knob.getSpin(0))
  { rotary_number += spin;
    if (rotary_number < 0) rotary_number = 9999;
    if (mode == MODE_CLOCK || mode == MODE_DIST)
         led7x4_Popup(rotary_number);
    else led7x4_Display(rotary_number);
    relayCommand( (rotary_number & 1) ? RELAY_ON:RELAY_OFF );
  }
  if (knob.getPress(0))
  { rotary_number = 0;
    if (mode == MODE_CLOCK || mode == MODE_DIST)
         led7x4_Popup(rotary_number);
    else led7x4_Display(rotary_number);
  }

  // IR receiver
  if (IrReceiver.decode()) 
  {
    beep(50);
    if (mode == MODE_CLOCK || mode == MODE_DIST)
         led7x4_Popup(IrReceiver.decodedIRData.command);
    else led7x4_Display(IrReceiver.decodedIRData.command);
    relayCommand( (IrReceiver.decodedIRData.command & 1) ? RELAY_ON:RELAY_OFF );
    String s = String(IrReceiver.decodedIRData.command);
    lcdPopup("IR",s.c_str());
    IrReceiver.resume();
  }

  // joystick
  { // 0..511..1023  dead zone 0..461/561..1023
    #define JOY_DEAD 50    
    
    // position a dot on LED matrix
    byte x = map( joystick_x, 0,1023, 0,7 );
    byte y = map( joystick_y, 0,1023, 0,7 );
    memset( matrix_buffer, 0, 8 );
    matrix_buffer[y] |= 1 << x;

    // up/down controls fan motor
    if (joystick_y > 511+JOY_DEAD)
      fan_control = 0x80 + map( joystick_y, 511+JOY_DEAD,1023, 0,63 );
    else if (joystick_y < 511-JOY_DEAD)
      fan_control = 0xC0 + map( joystick_y, 511-JOY_DEAD,0, 0,63 );
    else fan_control = 0;

    // right/left controls stepper motor
    if (joystick_x > 511+JOY_DEAD)
      fan_control = 0x80 + map( joystick_x, 511+JOY_DEAD,1023, 0,63 );
    else if (joystick_x < 511-JOY_DEAD)
      stepper_control = 0xC0 + map( joystick_x, 511-JOY_DEAD,0, 0,63 );
    else stepper_control = 0;
  }
  
  // build outputs for mode
  switch (mode)
  {
    case MODE_CLOCK: // from rtc_now
      // LCD
      // 1234567890123456
      //     mm/dd/yy
      //     hh:mm:ss
      memset( lcd_buffer, ' ', 2*16 );
      lcd_formatInt( 0, 4,rtc_now.month(),2 );
      lcd_formatInt( 0, 4,rtc_now.day(),2 );
      lcd_formatInt( 0, 4,rtc_now.year() % 100,2 );
      lcd_buffer[6] = lcd_buffer[9] = '/';
      lcd_formatInt( 1, 4,rtc_now.hour(),2 );
      lcd_formatInt( 1, 7,rtc_now.minute(),2 );
      lcd_formatInt( 1,10,rtc_now.second(),2 );
      lcd_buffer[16+6] = lcd_buffer[16+9] = ':';
      // 4 digit 7seg
      // hhmm
      led7x4_Display( 100*rtc_now.hour() + rtc_now.minute() );
      break;
    
    case MODE_ENV  :
      // LCD
      // 1234567890123456
      //  ##F ##F ###%RH
      //  ###   ### ###
      // Rain ### LDR ###
      memset( lcd_buffer, ' ', 2*16 );
      lcd_formatInt( 0,1,ntc_F,2 );
      lcd_formatInt( 0,5,dht_F,2 );
      lcd_buffer[3] = lcd_buffer[7] = 'F';
      lcd_formatInt(0,9,dht_RH,3);
      lcd_buffer[13] = 'R'; lcd_buffer[14] = 'H';
      lcd_formatInt( 1,1,rain_amount,3 );
      lcd_formatInt( 1, 7,ldr1_amount,3 );
      lcd_formatInt( 1,11,ldr2_amount,3 );
      break;    

    case MODE_SOUND:
      // LCD
      // 1234567890123456
      //   Sound Level
      memset( lcd_buffer, ' ', 2*16 );
      memcpy( lcd_buffer+2, "Sound Level", 11 );
      { uint16_t x = map( sound_amount, 0,1023, 0,16 );
        for (byte i=0; i < x; i++) lcd_buffer[16+i] = '>';
      }
      break;    
    
    case MODE_DIST : // from distance_cm
      // LCD
      // 1234567890123456
      //     Distance
      //     ##### cm
      memset( lcd_buffer, ' ', 2*16 );
      memcpy( lcd_buffer+4, "Distance", 8 );
      { String s = String(distance_cm);
        for (byte i=0; i < s.length(); i++) lcd_buffer[16+9-s.length()+i] = s.charAt(i)+'0';
        memcpy( lcd_buffer+16+10, "cm", 2 );
      }
      // 4 digit 7seg
      led7x4_Display( distance_cm );
      break;
    
    case MODE_IMU  :
      // LCD
      // 1234567890123456
      //  #### #### ####
      memset( lcd_buffer, ' ', 2*16 );
      memcpy( lcd_buffer+4, "Accelerometer", 13 );
      lcd_formatFloat( 1, 1,imu_accel.XAxis );
      lcd_formatFloat( 1, 6,imu_accel.YAxis);
      lcd_formatFloat( 1,11,imu_accel.ZAxis );
  }
  
//-- send outputs

  lcdTask();          // display from lcd_buffer
  led7x1_Task();      // manage single digit 7seg
  led7x4_Task();      // manage four digit 7seg
  matrixTask();       // update LED matrix from matrix_buffer
  srAnimationTask();  // run '595 LED animation
  cpAnimationTask();  // run charliplexed LED animation
  servoTask();        // manage servo
  fanTask();          // manage fan
  stepperTask();      // run stepper motor
  relayTask();        // manage relay
  PIRTask();          // PIR wake up animations
  tonePlayerTask();   // run melody player
  beepTask();         // run the beeper
}
