//-------------------------------------------------------------
// rotary encoder

#pragma once
#ifndef EncoderSW_lib
#define EncoderSW_lib
    
class EncoderSW // rotary encoder with switch
{
  public:

    enum rotor_accelleration_profile
    { ROTOR_ACCEL_NONE = 0,
      ROTOR_ACCEL_LOW  = 1,
      ROTOR_ACCEL_MED  = 2,
      ROTOR_ACCEL_MAX  = 3
    };

    enum button_event_code
    { BTN_PRESS   = 0,
      BTN_RELEASE = 1,
      BTN_HOLD    = 2    // long press
    };
        
  private:
  
    #define NUM_ENCODERS 1
    #define BUTTON_DEBOUNCE_MSEC 50
    #define HOLD_MSEC 1000 // long press time

    #define GPIO_DDR  DDRD
    #define GPIO_PORT PORTD
    #define GPIO_PINS PIND
    const byte pins[NUM_ENCODERS*3] =
    { // A  B  BTN
         2, 3, 7  // port bits
    };
    
   // VelociRotor acceleration profiles
    const struct
    { byte gear1_ms;
      byte gear2_ms;
      byte gear1_step;
      byte gear2_step;
      byte gear3_step;
    } profile[4] =
    {
      {   1,1,  1,1,1 }, // no velocirotor acceleration
      { 100,20, 1,2,4 }, // low acceleration
      { 100,20, 1,2,8 }, // medium acceleration
      { 100,10, 1,2,15}  // maximum acceleration
    };
    
    // rotary encoder state (one rotary encoder, not including button)
    struct encoder_t
    { rotor_accelleration_profile accel_profile;
      boolean  first;   // first-spin null zone suppression
      byte     phase;   // quadurature phase (0..3)
      int8_t   delta;   // phase delta accumulator
      uint32_t spin_tm; // time of last rotation, for velocirotor
      int8_t   spin;    // unreported rotation
      void (*spin_handler)(uint8_t index, int8_t spin);
      uint32_t spin_refresh; // max spin callback rate, msecs
      uint32_t spin_refresh_tm;
      boolean  btn;     // follows button state, debounced
      uint32_t btn_tm;  // button debounce timer
      uint32_t hold_tm; // long press timer
      boolean  hold_sent;
      boolean  pressed; // unreported button press
      uint8_t  button_mask; // disabled button events
      void (*button_handler)(uint8_t index, uint8_t event);
    } encoder[NUM_ENCODERS];
    
    uint8_t gpio_prev; // track GPIO state

  public:

    void begin()
    {
      for (uint8_t i=0; i < NUM_ENCODERS; i++)
      { encoder[i].first = true;
        encoder[i].phase = 0;
        encoder[i].delta = 0;
        encoder[i].spin  = 0;
        encoder[i].spin_tm = 0;
        encoder[i].spin  = 0;
        encoder[i].spin_handler  = NULL;
        encoder[i].spin_refresh_tm = millis();
        encoder[i].btn     = false;
        encoder[i].pressed = false;
        encoder[i].accel_profile = ROTOR_ACCEL_NONE;
        encoder[i].button_handler = NULL;
        encoder[i].button_mask = 0;
        GPIO_DDR  &= ~((1 << pins[0]) | (1 << pins[1]) | (1 <<pins[2] )); // inputs
        GPIO_PORT |=   (1 << pins[0]) | (1 << pins[1]) | (1 <<pins[2] );  // pullups
      }
//      gpio_prev = gpio_get_all(); // get initial state
      gpio_prev = GPIO_PORT;
    }

  //-- callbacks for rotor spin & button press
  //   NOT multiprocessor safe

    void setButtonHandler( uint8_t index, void (*button_handler)(uint8_t index, uint8_t event), uint8_t button_enable_mask )
    {
      if (index >= NUM_ENCODERS) return; // failsafe
      encoder[index].button_handler = button_handler;
      encoder[index].button_mask    = button_enable_mask;
    }

    void setSpinHandler( uint8_t index, void (*spin_handler)(uint8_t index, int8_t spin), uint32_t refresh_rate_msec )
    {
      if (index >= NUM_ENCODERS) return; // failsafe
      encoder[index].spin_handler = spin_handler;
      encoder[index].spin_refresh = refresh_rate_msec;
    }

  //-- alternate access to spin & push events

    int8_t getSpin( uint8_t index )
    {
      if (index >= NUM_ENCODERS) return 0; // failsafe
      int8_t result       = encoder[index].spin;
      encoder[index].spin = 0;
      return result;
    }

    boolean getPress( uint8_t index )
    {
      if (index >= NUM_ENCODERS) return false; // failsafe
      boolean result         = encoder[index].pressed;
      encoder[index].pressed = false;
      return result;
    }
    
  //-- encoder & button monitor

    void loop() // poll at 1msec (or faster) rate
    {
        uint8_t gpio_bits = GPIO_PINS;
        for (uint8_t i=0; i < NUM_ENCODERS; i++) 
        { //handleEncoder(i, gpio_bits);
          uint8_t A_mask = (1 << pins[i*3+0]);
          uint8_t B_mask = (1 << pins[i*3+1]);
          // detect change in A signal
          if ((gpio_bits & A_mask) != (gpio_prev & A_mask))
              switch (encoder[i].phase)
              { case 0: case 1: encoder[i].phase = (gpio_bits & A_mask) ? 0:1;  encoder[i].delta += (encoder[i].phase == 0) ?  1:-1;  break;
                case 2: case 3: encoder[i].phase = (gpio_bits & A_mask) ? 3:2;  encoder[i].delta += (encoder[i].phase == 2) ?  1:-1; 
              }
          // detect change in B signal
          if ((gpio_bits & B_mask) != (gpio_prev & B_mask))
              switch (encoder[i].phase)
              { case 0: case 3: encoder[i].phase = (gpio_bits & B_mask) ? 0:3;  encoder[i].delta += (encoder[i].phase == 0) ? -1: 1;  break;
                case 1: case 2: encoder[i].phase = (gpio_bits & B_mask) ? 1:2;  encoder[i].delta += (encoder[i].phase == 1) ?  1:-1;  
              }
          // detect full quadrature cycle completed (one detent rotation)
          if (encoder[i].phase == 0) // quadrature sync lock (prefents dead zone)
          { if (encoder[i].delta > (encoder[i].first ? 0:2))
            { uint32_t dt = millis() - encoder[i].spin_tm;
              uint8_t d;
              if      (dt < profile[encoder[i].accel_profile].gear2_ms) d = profile[encoder[i].accel_profile].gear3_step;
              else if (dt < profile[encoder[i].accel_profile].gear1_ms) d = profile[encoder[i].accel_profile].gear2_step;
              else                                                      d = profile[encoder[i].accel_profile].gear1_step;
              if ((int)encoder[i].spin - d < -127) encoder[i].spin = -127; else encoder[i].spin -= d;
              encoder[i].first = false;
              encoder[i].delta = 0;
              encoder[i].spin_tm = millis();
            }
            else if (encoder[i].delta < (encoder[i].first ? 0:-2))
            { uint32_t dt = millis() - encoder[i].spin_tm;
              uint8_t d;
              if      (dt < profile[encoder[i].accel_profile].gear2_ms) d = profile[encoder[i].accel_profile].gear3_step;
              else if (dt < profile[encoder[i].accel_profile].gear1_ms) d = profile[encoder[i].accel_profile].gear2_step;
              else                                                      d = profile[encoder[i].accel_profile].gear1_step;
              if ((int)encoder[i].spin + d > 127) encoder[i].spin = 127; else encoder[i].spin += d;
              encoder[i].first = false;
              encoder[i].delta = 0;
              encoder[i].spin_tm = millis();
            }
          }
          // send spin via callback
          if (encoder[i].spin)
            if (encoder[i].spin_handler)
              if (millis() - encoder[i].spin_refresh_tm >= encoder[i].spin_refresh)
              { (*encoder[i].spin_handler)(i,encoder[i].spin );
                  encoder[i].spin = 0;
                  encoder[i].spin_refresh_tm = millis();
              }
          // debounce button
          uint8_t BTN_mask = (1 << pins[i*3+2]);
          if ((gpio_bits & BTN_mask) == 0) // trigger press on leading edge of bounce
          {
            if (!encoder[i].btn) // not already pressed ?
            { // press event
              encoder[i].btn = true;
              encoder[i].hold_tm = millis();
              encoder[i].hold_sent = false;
              encoder[i].pressed = true;             
              if ((encoder[i].button_mask & (1 << BTN_PRESS)) == 0) // event not disabled ?
                if (encoder[i].button_handler) (*encoder[i].button_handler)(i,BTN_PRESS);
            }
            encoder[i].btn_tm = millis(); // [re]start debounce
            if (!encoder[i].hold_sent)
              if (millis() - encoder[i].hold_tm >= HOLD_MSEC)
              { // long press event
                encoder[i].hold_sent = true;
                if ((encoder[i].button_mask & (1 << BTN_HOLD)) == 0) // event not disabled ?
                  if (encoder[i].button_handler) (*encoder[i].button_handler)(i,BTN_HOLD);
              }
          }
          else // not pressed
            if (encoder[i].btn) // was pressed ?
              if (millis() - encoder[i].btn_tm > BUTTON_DEBOUNCE_MSEC)
              { // release event
                encoder[i].btn = false;
                if ((encoder[i].button_mask & (1 << BTN_RELEASE)) == 0) // event not disabled ?
                  if (encoder[i].button_handler) (*encoder[i].button_handler)(i,BTN_RELEASE);
              }              
        }
        gpio_prev = gpio_bits; // track signal state
      }
};

#endif
