//------------------------------------------------------
// Filament Clock 2025 (monochrome clock)
// 2025.11.25 RSP
// Target: ATmega328PB 5V/8MHz(CRYTAL) using Pololu A-Star board
//         AiP650E (CAI-219) clock display (monochrome, evil I2C)
//         NEO-6M timebase
//         LDR for automatic brightness
//         (12) LED filaments
//         (60) edge light WS2812 string
//
// User interface:
//    MODE switch: selects UTC offset OR brightness control 
//    UP/DN buttons: change value
//    Colon indicates GPS status
//      missing = no GPS signal
//      blinking = no GPS lock (doesn't happen with the newer NEO models)
//      solid = GPS locked
//------------------------------------------------------

// digital module color
#define __RED 0
#define __BLU 1
#define __GRN 2
uint8_t CLOCK_TYPE = __BLU;

//------------------------------------------------------
// coroutine macros

#define coBegin { switch(_state_) { case 0:;
#define coEnd        _state_ = 0; }}
#define coDelay(msec) { _state_ = __LINE__; _tm_=millis(); return; case __LINE__: if (millis()-_tm_ < msec) return; }

//-------------------------------------------------------------
// buttons

const uint8_t BTN_PIN[2] = { A6, SDA1 }; // PE0
const uint8_t SW_PIN = SCL1; // PE1

//------------------------------------------------------------
// GPS Neo-6M

#include "gps.h" // brings in time.h
gps GPS;
gps_time clocktime; // display time

//------------------------------------------------------
// EEPROM 

#define AUTOSAVE_MSEC 5000UL

#define MIN_BRIGHTNESS -7 // manual brightness adjust limits
#define MAX_BRIGHTNESS  7

#include <EEPROM.h>

struct configData
{ int8_t  utc_offset;  // -12/+14
  int8_t  brightness_manual_adj;  // MIN_BRIGHTNESS..MAX_BRIGHTNESS
};
struct configData config;
struct configData config_eeprom_image;

// wear leveled storage
//   ATmega328 FLASH has 1K total, pagesize = 1 byte
struct EEpage
{ byte polarity;
  struct configData config;
};
#define PAGE_SIZE (1+2)
#define NUM_PAGES 341 // to fit in 1024 bytes of EEPROM

// wear leveling logic
//   current data is indicated by change in polarity
//   when writing
//      if next cell polarity is wrong, fix it.
//   new EEPROM, all FF's
//      search finds no polarity change
//        use last data, flip polarity on next write (@0)
//   no need to initialize new EEPROM

uint16_t locateCurrentData()
{
  byte p = EEPROM.read(0); // pre-read 1'st polarity byte
  for (uint16_t i=1; i < NUM_PAGES; i++)
  { byte np = EEPROM.read(i*PAGE_SIZE);
    if ((p ^ np) & 0x80) // polarity is high bit
      return i-1;
  }
  return NUM_PAGES-1; // polarity change not found, data is last element 
}

void loadConfig()
{
  uint16_t block_index = locateCurrentData();
  EEPROM.get( block_index*PAGE_SIZE+1, config ); // skip polarity byte
}

void storeConfig()
{
  uint16_t block_index = locateCurrentData();
  byte p = EEPROM.read( block_index*PAGE_SIZE ) & 0x80; // get current polarity
  // store next page
  block_index = (block_index+1) % NUM_PAGES;
  if (block_index == 0) p ^= 0x80; // wrap, flip polarity
  EEPROM.write( block_index*PAGE_SIZE, p);
  EEPROM.put( block_index*PAGE_SIZE+1, config );
  // fix next polarity if necessary
  block_index = (block_index+1) % NUM_PAGES;
  byte np = EEPROM.read( block_index*PAGE_SIZE ) & 0x80;
  if (np == p) EEPROM.write( block_index*PAGE_SIZE, p ^ 0x80 );
}

//-------------------------------------------------------------
// automatic brightness

#define BRIGHTNESS_PIN A0 // LDR
uint8_t brightness = 0;   // 0..7

void brightnessTask()
{ static int _state_ = 0;  static uint32_t _tm_;
  static uint32_t total = 0; // running average
  static byte     count;

  coBegin
    for (count=0; count < 100; count++) // sample for 1 second 
    { coDelay(10)
      total -= total/100;
      total += analogRead( BRIGHTNESS_PIN );
    }
    int32_t v = total / 100; // calculate average
    int adc = constrain( v, 25, 750 );
    int br  = map( adc, 25,750, 0,7 );
    brightness = constrain( br, 0,7 );
    updateDeviceBrightness();
  coEnd
}

//-------------------------------------------------------------
// WS2812 LED strings

uint8_t adjustedBrightness( uint8_t b )
{ // apply manual brightness adjustment to sensor value
  int8_t nb = b + config.brightness_manual_adj;
  nb = constrain( nb, 0,7 );
  return nb;
}


#include <Adafruit_NeoPixel.h>
Adafruit_NeoPixel ss_pixels(60, 6, NEO_RGB + NEO_KHZ800);
Adafruit_NeoPixel hh_pixels( 4, 7, NEO_RGB + NEO_KHZ800);

const uint8_t SS_brightness[8] =
{ 1, 2, 3, 4, 5, 6, 7, 10 };

const uint8_t HH_brightness[8] =
{ 3, 5, 10, 15, 20, 30, 40, 50 };

void displaySSstring( byte br )
{
  for (uint8_t i=0; i < 60; i++)
  {
    uint8_t b = SS_brightness[br];
    uint32_t c = 0;
    if (i == clocktime.ss)
      switch (CLOCK_TYPE)
      { case __RED: c = hh_pixels.Color(b,0,0); break;
        case __GRN: c = hh_pixels.Color(0,b,0); break;
        case __BLU: c = hh_pixels.Color(0,0,b); break;
      }
    ss_pixels.setPixelColor( ((59-i)+28) % 60, c );
  }
  ss_pixels.show();
}

void displayHHstring( byte br )
{
  for (uint8_t i=0; i < 4; i++)
  { // 3 filaments on each WS2811
    int8_t filament_index = (clocktime.hh % 12) - 1;
    if (filament_index < 0) filament_index = 11;
    uint8_t c1 = (i*3+0 <= filament_index) ? HH_brightness[br] : 0;
    uint8_t c2 = (i*3+1 <= filament_index) ? HH_brightness[br] : 0;
    uint8_t c3 = (i*3+2 <= filament_index) ? HH_brightness[br] : 0;
    if (i == 3) c3 = HH_brightness[br]; // 12 filament is always on
    hh_pixels.setPixelColor( i, hh_pixels.Color(c1,c2,c3) );
  }
  hh_pixels.show();
}

//------------------------------------------------------
// flash signal generator

class coFlasher
{
  public:
    int _state_ = 0;  uint32_t _tm_;
    boolean flash;
    void loop()
    { coBegin
        coDelay( 500 )
        flash = !flash;
      coEnd
    }
};
coFlasher flashTask;

enum lamp_state { off, on, blink };

//------------------------------------------------------
// CAI 219 Display (AiP650E)

#define AIP650_INSTRUCTION 0x48
#define AIP650_RAM         0x68 // 6A 6C 6E => digits 1,2,3,4
#define AIP650_GETKEY      0x49

#define AIP650_CLK_PIN A5 // bit-banged "I2c"
#define AIP650_DIO_PIN A4

class AiP650E_ClockDisplay
{
  private:
    byte current_brightness = 0;
    byte ram_shadow[4];
    const byte SEG_TABLE[16] =
    { // !GFEDCBA
        0b0111111, //  0
        0b0000110, //  1
        0b1011011, //  2
        0b1001111, //  3
        0b1100110, //  4
        0b1101101, //  5
        0b1111101, //  6
        0b0000111, //  7
        0b1111111, //  8
        0b1101111, //  9
        0b1110111, // 10 A
        0b1111100, // 11 b
        0b0111001, // 12 C
        0b1011110, // 13 d
        0b1111001, // 14 E
        0b1110001  // 15 F
    };
        
  public:
    byte hh_bcd = 0;
    byte mm_bcd = 0;
    lamp_state hh    = on;
    lamp_state mm    = on;
    lamp_state pm    = off; // B7 of digit 1
    lamp_state colon = off; // B7 of digit 2
    lamp_state dash  = off; // B7 of digit 3
    lamp_state deg   = off; // B7 of digit 4

    int AiPsend( byte command, byte data )
    {
      // set start condition (DIO falling while CLK is high)
      pinMode( AIP650_DIO_PIN, OUTPUT ); // DIO low   
      // send command byte
      for (int8_t b=7; b >= 0; b--) // shift out MSB first
      { pinMode( AIP650_CLK_PIN, OUTPUT ); // CLK low
        pinMode( AIP650_DIO_PIN, (command & (1 << b)) ? INPUT:OUTPUT );
        pinMode( AIP650_CLK_PIN, INPUT ); // CLK high clocks in data
      }
      // AiP should send ACK=0 on 9'th clock
      pinMode( AIP650_CLK_PIN, OUTPUT ); // CLK low
      pinMode( AIP650_DIO_PIN, INPUT );
      pinMode( AIP650_CLK_PIN, INPUT ); // CLK HIGH
      if (digitalRead( AIP650_DIO_PIN )) return -1; // no command ack
      // send data byte
      for (int8_t b=7; b >= 0; b--) // shift out MSB first
      { pinMode( AIP650_CLK_PIN, OUTPUT ); // CLK low
        pinMode( AIP650_DIO_PIN, (data & (1 << b)) ? INPUT:OUTPUT );
        pinMode( AIP650_CLK_PIN, INPUT ); // CLK high clocks in data
      }
      // AiP should send ACK=1 on 9'th clock
      pinMode( AIP650_CLK_PIN, OUTPUT ); // CLK low
      pinMode( AIP650_DIO_PIN, INPUT );
      pinMode( AIP650_CLK_PIN, INPUT ); // CLK HIGH
      // set end condition
      pinMode( AIP650_CLK_PIN, OUTPUT ); // CLK low
      pinMode( AIP650_DIO_PIN, OUTPUT ); // DIO low
      pinMode( AIP650_CLK_PIN, INPUT ); // CLK HIGH
      pinMode( AIP650_DIO_PIN, INPUT ); // DIO HIGH
      return 0; // success
    }
    
    byte renderLamp( lamp_state l )
    {
      switch (l)
      { case off:   return 0;
        case on:    return 0x80;
        case blink: return ((millis() % 1000) > 500) ? 0x80 : 0;
      }
      return 0;
    }
    void renderTime( lamp_state l, byte *result, byte bcd )
    {
      if (l == on || (l == blink && flashTask.flash))
      { *(result++) = SEG_TABLE[ bcd >> 4 ];
        *(result)   = SEG_TABLE[ bcd & 15 ];
      }
      else { (*result++) = 0; (*result) = 0; }
    }
    
    int loop()
    { flashTask.loop(); // maintain the flasher
      // render display
      byte render[4];
      { renderTime( hh, &render[0], hh_bcd );
        renderTime( mm, &render[2], mm_bcd );
        if (hh_bcd < 0x10) render[0] &= 0; // leading zero blank
        render[0] |= renderLamp( pm );
        render[1] |= renderLamp( colon );
        render[2] |= renderLamp( dash );
        render[3] |= renderLamp( deg );
      }
      // update as necessary
      for (byte i=0; i < 4; i++)
        if (render[i] != ram_shadow[i])
#ifdef USE_HW_I2C
        { Wire.beginTransmission((AIP650_RAM+i*2) >> 1);
            Wire.write( ram_shadow[i] = render[i] );
          int e; if ((e = Wire.endTransmission())) return e;
        }
#else        
        { int e; e = AiPsend( AIP650_RAM+i*2, ram_shadow[i] = render[i] ); if (e) return e; }
#endif        
      return 0; // success
    }
  
    void blank() // turn off all LEDs
    {
      for (byte i=0; i < 4; i++) ram_shadow[i] = 0;
      pm = colon = dash = deg = off;
    }
    
    int brightness( byte level ) // 0..7 = min..max (always on though)
    { // map level 0..7 to AiP650E brightness level 1,2,3,4,5,6,7,0
      level &= 7;
      level = (level == 7) ? 0 : (level+1);
      if (level == current_brightness) return 0;
      current_brightness = level;
#ifdef USE_HW_I2C
      Wire.beginTransmission(AIP650_INSTRUCTION >> 1);
        Wire.write( (byte) 0x01 + (level << 4) );
      return Wire.endTransmission();
#else
      return AiPsend( AIP650_INSTRUCTION, 0x01 + (level << 4) );
#endif
    }

    int begin()
    { 
#ifdef USE_HW_I2C
      Wire.begin(); 
#endif      
      // zero the display RAM
      for (byte i=0; i < 4; i++)
#ifdef USE_HW_I2C
      { Wire.beginTransmission((AIP650_RAM+i*2) >> 1);
          Wire.write( ram_shadow[i] = (byte) 0 );
        int e; if ((e = Wire.endTransmission())) return e;
      }
#else
      AiPsend( AIP650_RAM+i*2, ram_shadow[i] = 0 );
#endif      
      // turn on display
#ifdef USE_HW_I2C
      Wire.beginTransmission(AIP650_INSTRUCTION >> 1);
        Wire.write( (byte) 0x11 ); // display on, sleep disable, 8 segment mode, brightness 1 (min)
      return Wire.endTransmission();
#else
      return AiPsend( AIP650_INSTRUCTION, 0x11 );
#endif      
    }
};
// this chews up the entire I2C bus
AiP650E_ClockDisplay display_mono;

//------------------------------------------------------
// Helpers

void updateDeviceBrightness()
{
  displaySSstring( adjustedBrightness(brightness) );
  displayHHstring( adjustedBrightness(brightness) );
  display_mono.brightness( adjustedBrightness(brightness) );
}

void adjustClock( int hh_add ) // add/subtract hours from clocktime
{
  int n = clocktime.hh + hh_add;
  if (n > 23) 
    clocktime.hh = n-24;
  else if (n < 0)
    clocktime.hh = n + 24;
  else clocktime.hh = n;
}

uint8_t binToBCD( uint8_t bin )
{
  return ((bin/10) << 4) + (bin % 10);
}

void handlePress( )
{
  if (digitalRead( SW_PIN )) // mode select
  { // set UTC offset
    if (!digitalRead( BTN_PIN[0] )) // up button pressed ? 
      if (config.utc_offset < 14) 
      { config.utc_offset++;
        adjustClock( 1 ); // bump clocktime
      }    
    
    if (!digitalRead( BTN_PIN[1] )) // down button pressed ? 
      if (config.utc_offset > -12) 
      { config.utc_offset--;
        adjustClock( -1 ); // bump clocktime
      }
  }
  else
  { // set brightness
    int8_t spin = 0;
    if (!digitalRead( BTN_PIN[0] )) // up button pressed ? 
      spin = 1;
    if (!digitalRead( BTN_PIN[1] )) // down button pressed ? 
      spin = -1;
    if (spin)
    { config.brightness_manual_adj += spin;
      config.brightness_manual_adj = constrain(config.brightness_manual_adj, -7, 7);
      updateDeviceBrightness();
    }
  }
}

//------------------------------------------------------

void setup() 
{
  // diagnostics
  pinMode(2,INPUT_PULLUP); // diagnostic pin - changes clock color

  // Neo-6M on Serial
  //   using receive only
  Serial1.begin(9600);    

  // WS2812 strings
  hh_pixels.begin();
  hh_pixels.clear();  
  hh_pixels.show();
  ss_pixels.begin();
  ss_pixels.clear();  
  ss_pixels.show();

  // load setings from EEPROM
  loadConfig();
  config_eeprom_image = config;
  if (config.brightness_manual_adj < -7 || config.brightness_manual_adj >  7) config.brightness_manual_adj = 0;
  if (config.utc_offset           < -12 || config.utc_offset            > 14) config.utc_offset = 0;
  
  // buttons
  pinMode( SW_PIN, INPUT_PULLUP );
  for (uint8_t i=0; i < 2; i++)
    pinMode( BTN_PIN[i], INPUT_PULLUP);

  // set up monochrome display
  pinMode( AIP650_CLK_PIN, INPUT ); digitalWrite( AIP650_CLK_PIN, LOW );
  pinMode( AIP650_DIO_PIN, INPUT ); digitalWrite( AIP650_DIO_PIN, LOW );
  display_mono.begin();

  // display POST
  display_mono.colon  = lamp_state::on;
  display_mono.pm     = lamp_state::on;
  display_mono.dash   = lamp_state::on;
  display_mono.deg    = lamp_state::on;
  display_mono.hh_bcd = 0x88;
  display_mono.mm_bcd = 0x88;
  display_mono.brightness(4);
  display_mono.loop();
  delay(1000);
  display_mono.hh = lamp_state::off;
  display_mono.mm = lamp_state::off;
  display_mono.colon  = lamp_state::off;
  display_mono.pm     = lamp_state::off;
  display_mono.dash   = lamp_state::off;
  display_mono.deg    = lamp_state::off;
  display_mono.loop();
  delay(250);
  // filaments
  hh_pixels.clear();  
  hh_pixels.setPixelColor( 0, 0x800000 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 0, 0x008000 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 0, 0x000080 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 0, 0 );
  hh_pixels.setPixelColor( 1, 0x800000 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 1, 0x008000 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 1, 0x000080 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 1, 0 );
  hh_pixels.setPixelColor( 2, 0x800000 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 2, 0x008000 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 2, 0x000080 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 2, 0 );
  hh_pixels.setPixelColor( 3, 0x800000 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 3, 0x008000 );
  hh_pixels.show();
  delay(250);
  hh_pixels.setPixelColor( 3, 0x000080 );
  hh_pixels.show();
  delay(250);
  hh_pixels.show();
  delay(250);
  hh_pixels.clear();  
  hh_pixels.show();
  // seconds ring
  for (uint8_t i=0; i < 60; i++) ss_pixels.setPixelColor( i, 0x000002 );
  ss_pixels.show();
  delay(1000);
  for (uint8_t i=0; i < 60; i++) ss_pixels.setPixelColor( i, 0x000200 );
  ss_pixels.show();
  delay(1000);
  for (uint8_t i=0; i < 60; i++) ss_pixels.setPixelColor( i, 0x020000 );
  ss_pixels.show();
  delay(1000);
  ss_pixels.clear();
  ss_pixels.show();
  display_mono.colon  = lamp_state::off;
  display_mono.pm     = lamp_state::off;
  display_mono.dash   = lamp_state::off;
  display_mono.deg    = lamp_state::off;
  display_mono.hh = lamp_state::on;
  display_mono.mm = lamp_state::on;
}

// - - - - - - - - - - - - -

void loop(void) 
{
  static uint32_t tm_1sec = millis();

//-- maintain timebase
  
  { static uint8_t last_second = 99;
    // millis() clock is master
    uint32_t tm_now = millis();
    if (tm_now - tm_1sec >= 1000UL)
    { tm_1sec += 1000UL;
      // advance time 1 second
      gps_time t;
      memcpy( (byte *)&t, (byte *)&clocktime, sizeof(t) );
      t.ss++;
      if (t.ss > 59)
      { t.ss = 0;
        t.mm++;
        if (t.mm > 59)
        { t.mm = 0;
          t.hh++;
          if (t.hh > 23)
            t.hh = 0;
        }
      }
      memcpy( (byte *)&clocktime, (byte *)&t, sizeof(clocktime) );
    }
    // discipline master clock to GPS time
    { gps_time t;
      if (GPS.listen( & t )) // got a new time update from GPS ?
      { // in sync with RTC
        tm_1sec = tm_now;
        memcpy( (byte *)&clocktime, (byte *)&t, sizeof(clocktime) );
        // adjust UTC to local time
        adjustClock( config.utc_offset );
        // monitor GPS status
        { uint8_t last_status = 99;
          if (last_status != GPS.status)
          { last_status = GPS.status;
            switch (last_status)
            { case GPS_RECEIVING: display_mono.colon = lamp_state::blink;  break;
              case GPS_LOCKED   : display_mono.colon = lamp_state::on;     break;
              default           : display_mono.colon = lamp_state::off;    break;
            }
          }
        }
      }
    }

    if (last_second != clocktime.ss)
    { last_second = clocktime.ss;
      // up/down at 1per second rate, possibly update clocktime & utc_offset
      handlePress(); 
      // diagnostic: change clock color
      if (!digitalRead(2)) CLOCK_TYPE = (CLOCK_TYPE+1) % 3;
      // transfer hh:mm to bcd for digital display
      display_mono.hh_bcd = binToBCD( clocktime.hh > 12 ? clocktime.hh-12 : (clocktime.hh == 0 ? 12 : clocktime.hh) );
      display_mono.mm_bcd = binToBCD( clocktime.mm );
      display_mono.pm = (clocktime.hh >= 12) ? lamp_state::on : lamp_state::off;
    }
  }

//-- sample LDR, update brightness

  brightnessTask();

//-- update displays (when time changes)

  // digital display
  display_mono.loop();

  // filaments
  { static uint8_t last_hh = 99;
    if (last_hh != clocktime.hh)
    { last_hh = clocktime.hh;
      displayHHstring(adjustedBrightness(brightness));
    }
  }
  
  // seconds ring
  { static uint8_t last_ss = 99;
    if (last_ss != clocktime.ss)
    { last_ss = clocktime.ss;
      displaySSstring(adjustedBrightness(brightness));
    }
  }

//-- monitor for config changes

  { static struct configData config_copy;
    static boolean config_dirty = false;
    static uint32_t config_tm;
   
    if (memcmp( &config, &config_eeprom_image, sizeof(config) ) == 0)
      config_dirty = false;
    else if (!config_dirty)
    { config_dirty = true;
      config_copy = config;
      config_tm = millis();
    } 
    else if (memcmp( &config_copy, &config, sizeof(config) ))
    { config_copy = config;
      config_tm = millis();
    } 

    if (config_dirty) 
      if (millis() - config_tm > AUTOSAVE_MSEC)
      { config_dirty = false;
        config_eeprom_image = config;
        storeConfig(); // to EEPROM
      }
  }
}
