//-------------------------------------------------------------
// GPS time receiver
//  for Neo-6M

#include "gps.h"
#include "Arduino.h"

#define GPS_LOS_TIMEOUT_MS 5000 // 5 second GPS LOS window

// Ublox Neo-6M
//   0  1   2         3      4          5 6           7 8     9     10    11  12  13   14
//   $GPRMC,hhmmss,   status,latitude,  N,longitude,  E,spd,  cog,  ddmmyy,mv,mvE,mode*cs<CR><LF>
//   $GPRMC,083559.00,A,     4717.11437,N,00833.91522,E,0.004,77.52,091202,  ,   ,A   *57
//
//   $GPRMC,002740.00,A,     3339.90968,N,11210.28867,W,0.074,     ,040117,  ,   ,D   *62
//   $GPRMC,002740.00,A,3339.90968,N,11210.28867,W,0.074,,040117,,,D*62
//
//    Test messages
//        $GPTXT,01,01,02,MA=CASIC*27
//        $GPRMC,,V,,,,,,,,,,N*53

//      new format...
//        $GNRMC,061747.000,A,3339.91074,N,11210.29248,W,0.00,0.00,131225,,,A*60

//        $GPRMC,002740.00,A,3339.90968,N,11210.28867,W,0.074,,040117,,,D*62
//        $GPRMC,012740.00,A,3339.90968,N,11210.28867,W,0.074,,040117,,,D*63
//        $GPRMC,102740.00,A,3339.90968,N,11210.28867,W,0.074,,040117,,,D*63
//        $GPRMC,002740.00,A,3339.90968,N,11210.28867,W,0.074,,140117,,,D*63
//        $GPGSV,4,2,13,16,50,054,,22,23,177,41,23,76,047,30,26,23,043,27*7C
//        $GPGGA,002740.00,3339.90968,N,11210.28867,W,2,08,0.93,383.6,M,-27.7,M,,0000*6B
//        NOTE - Arduino serial monitor line limit!    

uint8_t gps::hex_to_bin( int h )
{
  if (h >= '0' && h <= '9')
    return h-'0';
  else if (h >= 'A' && h <= 'F') 
    return h-'A'+10; 
    else return 0;
}

uint8_t gps::asc2bin( char *dp )
{
  return 10*((*dp)-'0') + ((*(dp+1))-'0');
}

// receive GPS message ($...*CC)
boolean gps::listen( struct gps_time *t ) // returns true if t is populated with time from GPS
{
  do // until no more rx data is available
  {
    int ch = Serial1.read();
    if (ch == -1) break;
//Serial.write(ch);

    // got keepalive
    gps_ms = millis();

    // starting to receive
    if (status == GPS_LOS) status = GPS_RECEIVING;

    // always hunting for start of message
    if (ch == '$')
    { gps_state  = 1;
      gps_index  = 0;
      gps_chksum = 0;
      gps_is_valid_time = true;
      continue;
    }

    if (gps_state < 8 && ch != '*') gps_chksum ^= ch;

    if (gps_state > 1 && ch == '*') { gps_state = 8; gps_index = 0; continue; }

    switch (gps_state)
    {
      case 1: // receiving message type field
        // ignore message with invalid type field
        if (ch == ',')
        {
          if (gps_index == 5)
          { if (gps_date[0] == 'G'
             &&(gps_date[1] == 'N' || gps_date[1] == 'P')
             && gps_date[2] == 'R'
             && gps_date[3] == 'M'
             && gps_date[4] == 'C')
            { gps_state = 2; // good, continue
              gps_index = 0;
            }
            else // unwanted message, ignore it
              gps_state = 0; // skip to next message
          }            
          else // invalid header, ignore message
          { gps_state = 0; }
        }          
        else { if (gps_index < 5) gps_date[ gps_index++ ] = ch; }

        break;
        
      case 2: // receiving time field
        if (ch == ',')
        { if (gps_index >= 6)
            gps_state = 4;
          else // min 6 time digits
          { gps_state = 7;
            gps_is_valid_time = false; 
          }
        }
        else
        { if (gps_index <= 5) 
          { gps_time[ gps_index++ ] = ch;
            if (ch < '0' || ch > '9')  gps_is_valid_time = false;
          }          
        }
        break;

      case 4: // receiving status field
        if (ch == 'A')
        { gps_state = 5;
          gps_index = 0;
        }
        else
        { gps_state = 7;
          gps_is_valid_time = false; 
        }       
        break;

      case 5: // skipping to date field
        if (ch == ',')
        { if (gps_index == 6)
          { gps_state = 6;
            gps_index = 0;
          }
          else
            gps_index++;
        }
        break;
            
      case 6: // receiving date field
        if (ch == ',')
        { if (gps_index >= 6)
            gps_state = 7;
          else // min 6 date digits
          { gps_state = 7;
            gps_is_valid_time = false; 
          }
        }
        else
        { if (gps_index <= 5)
          { gps_date[ gps_index++ ] = ch;
            if (ch < '0' || ch > '9')  gps_is_valid_time = false;
          }
        }
        break;

      case 7: // skipping to end of message
        break;

      case 8: // collecting checksum
        if (gps_index == 0)
        { gps_temp = hex_to_bin( ch );
          gps_index = 1;
        }
        else
        { gps_temp = (gps_temp << 4) + hex_to_bin( ch );      
          // validate chksum
          if (gps_temp == gps_chksum)
          {
            if (gps_is_valid_time)
            { 
              // copyout gps time)
              t->hh  = asc2bin( gps_time+0 );
              t->mm  = asc2bin( gps_time+2 );
              t->ss  = asc2bin( gps_time+4 );
              t->dom = asc2bin( gps_date+0 );
              t->mon = asc2bin( gps_date+2 );
              t->yy  = asc2bin( gps_date+4 );
              status = GPS_LOCKED;
              gps_state = 0;
              return true;
            }
          }
          else // checksum error
          {
            status = GPS_ERROR;
          }

          // reset for next message
          gps_state = 0;
        }
    }

  } while (1);

  if (millis() - gps_ms > GPS_LOS_TIMEOUT_MS)
    status = GPS_LOS;

  return false;
}
