//-------------------------------------------------------------
// Gophert UI Hack 2020 
// Replacement front panel UI for NPS-1600/1601/1602
// 2020.09.01 RSP
//
// Board: Teensy 3.1/3.2
//   plus ADS1115 4 channel 15 bit ADC
//-------------------------------------------------------------

#define ENABLE_EE             // enable EE writes
#define ENABLE_TRACKING true  // enable voltage/current tracking

//------------------------------------------------------------
// 320x240 LCD ILI9341 & Touch Panel XPT2046

#include <SPI.h>

#include <ILI9341_t3.h>
#include <font_ArialBold.h> // from ILI9341_t3
#define TFT_DC  9
#define TFT_CS 10
ILI9341_t3 tft = ILI9341_t3(TFT_CS, TFT_DC);

#include <XPT2046_Touchscreen.h>
#define CS_PIN  15 // T_CS
#define TIRQ_PIN 2 // wired but not used
//XPT2046_Touchscreen ts(CS_PIN, TIRQ_PIN);  // Param 2 - Touch IRQ Pin - interrupt enabled polling
XPT2046_Touchscreen ts(CS_PIN);

#define BRIGHTNESS_DAC A14
#define DEFAULT_BRIGHTNESS 4
const uint8_t brightness_table[10] = { 48, 50, 52, 54, 56, 58, 60, 62, 66, 255 };

void setBrightness( int b )
{      
  analogWriteResolution( 8 );
  b = constrain( b, 0,9 );
  analogWrite(BRIGHTNESS_DAC, brightness_table[ b ]);
}

// icon button images
#include "power60.c" 
#include "presets60.c" 
#include "wrench60.c" 
#include "knob_icon.c"

//------------------------------------------------------------
// data structures

struct touch_cal_struct
{
  int16_t min_x; 
  int16_t max_x;
  int16_t min_y;
  int16_t max_y;   
};

// semi-permanent settings
//   64 bytes allocated (= 8 blocks)
//   @ 0
#define EE_SIGNATURE 0x12BE
#define NUMBLOCKS_EE_ADDR 2
struct config_data
{
  uint16_t signature;  // initialized EE detection
  uint8_t  num_blocks; // number of settings_data blocks
  uint8_t  brightness; // 0-9
  uint8_t  safe_limit; // 0=none, 1=3.3V, 2=5V, 3=12V
  touch_cal_struct touch_cal;
  uint8_t  model;      // NPS-160x, x=0,1,2
  uint16_t cv_cal[6];  // control calibration, adc counts
  uint16_t cc_cal[4];
  uint16_t vs_cal[6];  // sense calibration, adc counts
  uint16_t cs_cal[4];
};

const char *SafeLimitText( uint8_t code )
{
  switch (code)
  { case 1 : return "3.3";
    case 2 : return "5";
    case 3 : return "12";
    default: return "No Limit";
  }
}

// often updated settings (voltage & current)
//   8 byte block
//   @ 64 (0x0040)
//   x 248 blocks
//   100,000 write limit per block
//   = 24.8M settings updates
#define MAX_EE_BLOCKS   (256-8)
#define SETTINGS_EE_ADDR 64     // starting EEPROM address
struct settings_data
{ 
  uint32_t write_count; // 100001 if block is full
  uint16_t cv,cc;       // set points (V/A x100)
};

// EEPROM image
config_data   ee_config;
settings_data ee_settings;
// current settings data block index
int      ee_current_settings_block; // -1 if settings writes are exhausted
uint32_t ee_write_count;            // total writes

// live data (not saved to EEPROM yet)
config_data   ui_config;
settings_data ui_settings;

// UI state
#define MAX_BUTTONS  10 // on one screen
struct
{ int      mode;  // current screen
  uint16_t vs,cs; // V/A x100
  boolean  v_altcolor, c_altcolor;
  int      cccv;  // 0=no display, 1=CV, 2=CC
  boolean  whcs;  // A vs. W display
  int      temp;  // C
  uint16_t cv,cc; // V/A x100
  boolean  cv_display_pend;
  boolean  cc_display_pend;
  boolean  setting_cc; // cv vs. cc
  // buttons on the screen
  uint8_t  button_count;
  struct
  { uint16_t x,y;
    uint16_t w,h;
    char text[20];
  } button[ MAX_BUTTONS ];
  // auto-save update to EEPROM
  boolean  update_pending;
  uint32_t update_tm;
} ui;

// new EEPROM
boolean first_time_setup = false;

//------------------------------------------------------------
// control signals to PSU

// beware additional series resistor in VS line, different on each model
#define VS_DIVIDER_R1 39000
#define VS_DIVIDER_R2  4700
#define CS_DIVIDER_R1 10000
#define CS_DIVIDER_R2  2210

const struct
{ char     name[10];        // PSU model
  uint16_t max_v_x100;      // PSU limits
  uint16_t max_a_x100;
  // calibration points
  uint16_t v_cal_points[6]; // V x100
  uint16_t a_cal_points[4]; // A x100
  // safe limits
  char     limit_label[4][10];
  uint16_t limits[ 3 ];     // safe limits, V x100
  // presets
  char     preset_v_label[4][10];
  uint16_t preset_v[4];     // preset V x100
  char     preset_a_label[4][10];
  uint16_t preset_a[4];     // preset A x100
} psu_specs[ 3 ] =
{
  { "NPS-1600", 1600, 1000,
    { 0, 100, 300, 500, 1000, 1600 },
    { 0, 10, 100, 200 },
    { "3.3V", "5V", "12V" }, { 360, 550, 1350 },
    { "3.3V", "5V", "12V", "15V" }, { 330, 500, 1200, 1500 },
    { "100mA", "250mA", "1A", "2A" }, { 10, 25, 100, 200 }
  },
  { "NPS-1601", 3200, 500,
    { 0, 300, 500, 1000, 2000, 3200 },
    { 0, 10, 100, 200 },
    { "3.3V", "5V", "12V" }, { 360, 550, 1350 },
    { "3.3V", "5V", "12V", "24V" }, { 330, 500, 1200, 2400 },
    { "100mA", "200mA", "500mA", "1A" }, { 10, 20, 50, 100 }
  },
  { "NPS-1602", 6000, 300,
    { 0, 300, 500, 1000, 3000, 6000 },
    { 0, 10, 100, 200 },
    { "3.3V", "5V", "12V" }, { 360, 550, 1350 },
    { "3.3V", "5V", "12V", "24V" }, { 330, 500, 1200, 2400 },
    { "100mA", "200mA", "500mA", "1A" }, { 10, 20, 50, 100 }
  }
};
 
// CV & CC on PWM pins
#define PWM_CV_PIN 3
#define PWM_CC_PIN 4

// enable power pump
#define ENA_PUMP_PIN 20

#define MAX_TRACKING_CORRECTION 0.10 // percent

// PSU control state
struct
{ boolean  is_on;
  uint32_t pump_tm;
  boolean  pump_phase;
  float    vcc_3V3;
  uint16_t vs_adc;
  uint16_t cs_adc;
  int16_t cv_tracking_err_accumulator, cc_tracking_err_accumulator;
  float   cv_tracking_correction_pct,  cc_tracking_correction_pct;
} psu = { false, 0, false, 3.30, 0, 0, 0, 0, 0.0, 0.0 };

void startPSU()
{
  psu.is_on = true;
  psu.pump_phase = true;
  digitalWrite( ENA_PUMP_PIN, psu.pump_phase );
  psu.pump_tm = millis();
}

void stopPSU()
{
  psu.is_on = false;
  digitalWrite( ENA_PUMP_PIN, LOW );
}

void setCV( uint16_t cv_x100 )
{
  uint8_t i;
  for (i=1; i < 5; i++) // locate calibration bracket
    if (cv_x100 <= psu_specs[ ui_config.model ].v_cal_points[i]) break;
  uint16_t dac = map( cv_x100 + (ENABLE_TRACKING && psu.is_on ? cv_x100 * psu.cv_tracking_correction_pct : 0),
                      psu_specs[ ui_config.model ].v_cal_points[i-1], psu_specs[ ui_config.model ].v_cal_points[i],
                      ui_config.cv_cal[ i-1 ], ui_config.cv_cal[ i ] );
  analogWriteResolution( 15 );
  analogWrite( PWM_CV_PIN, dac );
  ui.cv = cv_x100;
}
void setCC( uint16_t cc_x100 )
{
  uint8_t i;
  for (i=1; i < 3; i++) // locate calibration bracket
    if (cc_x100 <= psu_specs[ ui_config.model ].a_cal_points[i]) break;
  uint16_t dac = map( cc_x100 + (ENABLE_TRACKING && psu.is_on ? cc_x100 * psu.cc_tracking_correction_pct : 0), 
                      psu_specs[ ui_config.model ].a_cal_points[i-1], psu_specs[ ui_config.model ].a_cal_points[i],
                      ui_config.cc_cal[ i-1 ], ui_config.cc_cal[ i ] );
  analogWriteResolution( 15 );
  analogWrite( PWM_CC_PIN, dac );
  ui.cc = cc_x100;
}

//------------------------------------------------------------
// rotary encoder (& button)

#define ENC_A_PIN 16
#define ENC_B_PIN 17
#define ENC_BUTTON_PIN 21

// 1 msec interrupt
IntervalTimer myTimer;

// rotary encoder
struct
{ volatile boolean first; // first-spin null zone suppression
  volatile byte    a,b;   // follows encoder A & B bits
  volatile byte    phase; // quadurature phase (0..3)
  volatile int8_t  delta; // phase delta accumulator  
  volatile uint32_t tick; // 1 msec counter
  volatile uint32_t last_tick;
  volatile int     value;
} encoder;

// rotary encoder button
#define LONG_PRESS_MS      1000
#define BUTTON_DEBOUNCE_MS   50
struct
{ volatile byte     b;
  volatile boolean  b_long;
  volatile uint32_t tm;
  volatile uint32_t long_press_tm;
  volatile boolean  pressed;
  volatile boolean  long_pressed;
} button;

void int1msec()
{

//-- button debouncing

  { byte b = digitalReadFast( ENC_BUTTON_PIN );
  
    if (!b) // pressed
    { if (!button.b)
      { // button press event
        button.pressed = true;
        button.b = 1;
        button.b_long = false;
        button.long_press_tm = millis();
      }
      button.tm = millis();
      if (!button.b_long) 
        if (millis() - button.long_press_tm >= LONG_PRESS_MS)
        { // long press event
          button.b_long = true; // one-shot button  
          button.long_pressed = true;
        }
    }
    else if (button.b) // debounce on release
      if (millis() - button.tm >= BUTTON_DEBOUNCE_MS)
        button.b = 0;
  }

//-- rotary encoder decoding 

  encoder.tick++; // free running 1 msec timer
  
  { byte a = digitalReadFast( ENC_A_PIN );
    if (a != encoder.a)
      switch (encoder.phase)
      { case 0: case 1: encoder.phase = a ? 0:1;  encoder.delta += (encoder.phase == 0) ?  1:-1;  break;
        case 2: case 3: encoder.phase = a ? 3:2;  encoder.delta += (encoder.phase == 2) ?  1:-1; 
      }
    
    byte b = digitalReadFast( ENC_B_PIN );
    if (b != encoder.b)
      switch (encoder.phase)
      { case 0: case 3: encoder.phase = b ? 0:3;  encoder.delta += (encoder.phase == 0) ? -1: 1;  break;
        case 1: case 2: encoder.phase = b ? 1:2;  encoder.delta += (encoder.phase == 1) ?  1:-1;  
      }
  
    encoder.a = a;
    encoder.b = b;
  
    if (encoder.phase == 0) // quadrature sync lock (prevents dead zone)
    {
      // 3 speed velocarotor tuning values
      const int gear_1_ms  = 250;
      const int gear_1_rot =   1;
      const int gear_2_ms  = 100;
      const int gear_2_rot =   2;
      const int gear_3_rot =  15;
      
      if (encoder.delta > (encoder.first ? 0:2))
      { encoder.delta = 0;
        encoder.first = false;
        uint32_t dt = encoder.tick - encoder.last_tick;
        int incr = (dt > gear_1_ms) ? gear_1_rot:(dt > gear_2_ms ? gear_2_rot:gear_3_rot);
        encoder.value -= incr;
        encoder.last_tick = encoder.tick;
      }
      else if (encoder.delta < (encoder.first ? 0:-2))
      { encoder.delta = 0;
        uint32_t dt = encoder.tick - encoder.last_tick;
        int incr = (dt > gear_1_ms) ? gear_1_rot:(dt > gear_2_ms ? gear_2_rot:gear_3_rot);
        encoder.value += incr;
        encoder.last_tick = encoder.tick;
      }
    }
  }
}

//------------------------------------------------------------
// status LED
//   blue   = disabled and < 1V
//   yellow = disabled and > 1V
//   red    = enabled and > 1V

#include <Adafruit_NeoPixel.h>
#define PIX_PIN 14
#define NUM_LEDS 1
Adafruit_NeoPixel status_pixel = Adafruit_NeoPixel(NUM_LEDS, PIX_PIN, NEO_RGB + NEO_KHZ800);

int status_led = -1;

void paint_status_led( int nstate )
{
  switch (status_led = nstate)
  {
    case 0 : status_pixel.setPixelColor( 0, status_pixel.Color( 0, 0, 2) );  break; // blue
    case 1 : status_pixel.setPixelColor( 0, status_pixel.Color(16,16, 0) );  break; // yellow
    default: status_pixel.setPixelColor( 0, status_pixel.Color(16, 0, 0) );  break; // red
  }
  status_pixel.show();
}

//------------------------------------------------------------
// settings in EEPROM
// Teensy 3.1/3.2 EEPROM is 2048 bytes; 100,000 write endurance (per cell)

#include <EEPROM.h>

void loadConfig()
{
  EEPROM.get( 0, ee_config );
  // validate config
  if (ee_config.signature != EE_SIGNATURE)
  { 
    // initialize EEPROM
    first_time_setup = true; // automatically prompt for touch & PSU calibration
    
    // show prompt screen
    //   "turn dial to select model"
    //   "press to initialize EEPROM"
    tft.fillScreen(ILI9341_BLACK);
    tft.setTextColor( ILI9341_WHITE );
    tft.setFont(Arial_14_Bold);
    tft.setCursor( 94,50 );
    tft.print( "Setup" ); 
    tft.setFont(Arial_8_Bold);
    tft.setCursor( 70,110 );
    tft.print( "Press to initialize" ); 
    // wait for button
    while (!button.pressed);
    button.pressed = false;

    // progress screen
    tft.fillScreen(ILI9341_BLACK);
    tft.setTextColor( ILI9341_WHITE );
    tft.setFont(Arial_14_Bold);
    tft.setCursor( 50,50 );
    tft.print( "Initializing..." ); 
    
    // build config
    ee_config.model      = 0; // NPS-1600
    ee_config.num_blocks = 0;
    ee_config.brightness = 2; // nominal brightness setting
    ee_config.safe_limit = 0; // no limit
    // default touch calibration
    ee_config.touch_cal.min_x =  344; 
    ee_config.touch_cal.min_y =  229; 
    ee_config.touch_cal.max_x = 3488; 
    ee_config.touch_cal.max_y = 3500; 
    // default PSU calibration
    ee_config.cv_cal[0]=9;    ee_config.cv_cal[1]=1516; ee_config.cv_cal[2]=2526;  ee_config.cv_cal[3]=5054; ee_config.cv_cal[4]=15125; ee_config.cv_cal[5]=30099; 
    ee_config.cc_cal[0]=5060; ee_config.cc_cal[1]=5962; ee_config.cc_cal[2]=14374; ee_config.cc_cal[3]=22995;
    ee_config.vs_cal[0]=30;   ee_config.vs_cal[1]=1468; ee_config.vs_cal[2]=2455;  ee_config.vs_cal[3]=4924; ee_config.vs_cal[4]=14770; ee_config.vs_cal[5]=29417; 
    ee_config.cs_cal[0]=3095; ee_config.cs_cal[1]=4750; ee_config.cs_cal[2]=11869; ee_config.cs_cal[3]=19609;
    // config is now valid, but no settings blocks allocated
    saveConfig(); 
    
    // zero all block write counts
    //   EEPROM.length() is unreliable for some EEPROM configurations
    //   so we just write blocks until it finishes or hangs
    ee_current_settings_block = 0;
    settings_data blk;
    blk.cv = blk.cc = 0;
    blk.write_count = 0;
    for (int i = 0; i < MAX_EE_BLOCKS; i++)
    {
      tft.fillRect( 100,110, 100,50, ILI9341_BLACK);
      tft.setFont(Arial_14_Bold);
      tft.setCursor( 100,110 );
      tft.print( String(i+1) ); 
      
      uint16_t addr = SETTINGS_EE_ADDR + i * sizeof(settings_data);
      // this will hang if invalid EE address
#ifdef ENABLE_EE    
      EEPROM.put( addr, blk );
#endif
      // update successful block count
      ee_config.num_blocks++;
#ifdef ENABLE_EE    
      EEPROM.put( NUMBLOCKS_EE_ADDR, ee_config.num_blocks );
#endif
    }
  }
  ee_config.model      = constrain( ee_config.model,      0,2 );
  ee_config.brightness = constrain( ee_config.brightness, 0,9 );
  ee_config.safe_limit = constrain( ee_config.safe_limit, 0,3 );
}

void saveConfig()
{
  ee_config.signature = EE_SIGNATURE;
#ifdef ENABLE_EE    
  EEPROM.put( 0, ee_config );
#endif  
}

void loadSettings()
{
  // find first block with a count < 100,001
  ee_write_count = 0; // count total writes used
  for (ee_current_settings_block = 0; ee_current_settings_block < ee_config.num_blocks; ee_current_settings_block++)
  {
#ifdef ENABLE_EE    
    EEPROM.get( SETTINGS_EE_ADDR + ee_current_settings_block * sizeof(settings_data), ee_settings );
#else
  ee_settings.cv = 0;
  ee_settings.cc = 0;
  ee_settings.write_count = 1;
#endif    
    if (ee_settings.write_count < 100001UL) { ee_write_count += ee_settings.write_count;  break; }
    ee_write_count += 100000;
  }
  if (ee_current_settings_block >= ee_config.num_blocks)
  { // all blocks are full!
    ee_settings.cc = ee_settings.cv = 0;
    ee_current_settings_block = -1; // disable save
  }
}

void saveSettings()
{
  if (ee_current_settings_block < 0) return; // can't save

  if (ee_settings.write_count == 100000UL)
  { // close block
    ee_settings.write_count++; // 100001
#ifdef ENABLE_EE    
    EEPROM.put( SETTINGS_EE_ADDR + ee_current_settings_block * sizeof(settings_data), ee_settings );
#endif    
    ee_settings.write_count = 0; // start new block
    ee_current_settings_block++;
    if (ee_current_settings_block >= ee_config.num_blocks) { ee_current_settings_block = -1;  return; }
  }
  ee_settings.write_count++; // track write count to this block
  ee_write_count++;          // track total write count
#ifdef ENABLE_EE  
  EEPROM.put( SETTINGS_EE_ADDR + ee_current_settings_block * sizeof(settings_data), ee_settings );  
#endif    
}

void saveChanges()
{
  // ui_config -> ee_config -> EEPROM
  if (memcmp( &ui_config, &ee_config, sizeof(config_data) ))
  { ee_config = ui_config;
    saveConfig(); // from ee_config
Serial.println("CONFIG UPDATED");          
  }
  // ui_settings -> ee_settings -> EEPROM
  if (memcmp( &ui_settings, &ee_settings, sizeof(settings_data) ))
  { ee_settings = ui_settings;
    saveSettings();
Serial.println("SETTINGS UPDATED");          
  }
}

//-------------------------------------------------------------
// ADS1115 A/D on I2C

#include "Wire.h"
#include "ADS1115.h"

static uint16_t readRegister16(uint8_t i2cAddress, uint8_t reg) 
{
  Wire.beginTransmission(i2cAddress);
  Wire.write((uint8_t)reg);
  Wire.endTransmission();
  Wire.requestFrom(i2cAddress, (uint8_t)2);
  return ((Wire.read() << 8) | Wire.read()); 
}

void writeRegister16(uint8_t i2cAddress, uint8_t reg, uint16_t value) 
{
  Wire.beginTransmission(i2cAddress);
  Wire.write((uint8_t)reg);
  Wire.write((uint8_t)(value>>8));
  Wire.write((uint8_t)(value & 0xFF));
  Wire.endTransmission();
}

boolean checkI2Cdevice( uint8_t addr )
{
  Wire.beginTransmission(addr);
  return (Wire.endTransmission() == 0);
}

//------------------------------------------------------------
// user interface

#define AUTOSAVE_MS 5000

#define DISP_H 320
#define DISP_W 240

#define CAL_TIMEOUT_MS 10000UL

#define BACKGND_COLOR ILI9341_BLACK
#define TEXT_COLOR    ILI9341_WHITE
#define TEMP_COLOR    ILI9341_PINK
#define LIMIT_COLOR   ILI9341_CYAN // same as set point color

#define SETTINGS_TEXT_COLOR ILI9341_GREEN
#define ERROR_TEXT_COLOR    ILI9341_RED
#define INFO_TEXT_COLOR     ILI9341_WHITE

#define BUTTON_COLOR  ILI9341_LIGHTGREY
#define BUTTON_HILITE ILI9341_ORANGE
#define BUTTON_TEXT   ILI9341_DARKGREEN
#define BUTTON_BORDER ILI9341_WHITE

// screen numbers
#define UI_MODE_MAIN           0
#define UI_MODE_POWER          1
#define UI_MODE_SETTINGS       2
#define UI_MODE_BRIGHTNESS     3
#define UI_MODE_LIMITS         4
#define UI_MODE_PRESETS        5
#define UI_MODE_MODEL          6
#define UI_MODE_PSU_CAL_V      7
#define UI_MODE_PSU_CAL_SAVE1  8
#define UI_MODE_PSU_CAL_A      9
#define UI_MODE_PSU_CAL_SAVE2 10
#define UI_MODE_INFO          11
#define UI_MODE_CAL_START     99
#define UI_MODE_CAL_POINT    100

uint16_t getLimitVx100( uint8_t limit_index )
{
  switch (limit_index)
  { case 0: return psu_specs[ ui_config.model ].max_v_x100;
    case 1: return psu_specs[ ui_config.model ].limits[0];
    case 2: return psu_specs[ ui_config.model ].limits[1];
    case 3: return psu_specs[ ui_config.model ].limits[2];
  }
  return 0; // failsafe
}

void addButton( uint16_t x, uint16_t y, uint16_t w, uint16_t h, const char *text )
{
  if (ui.button_count < MAX_BUTTONS)
  { ui.button[ui.button_count].x = x;
    ui.button[ui.button_count].y = y;
    ui.button[ui.button_count].w = w;
    ui.button[ui.button_count].h = h;
    strcpy( ui.button[ui.button_count].text, text );
    ui.button_count++;
  }
}

void addButton( uint16_t x, uint16_t y, uint16_t w, uint16_t h )
{
  if (ui.button_count < MAX_BUTTONS)
  { ui.button[ui.button_count].x = x;
    ui.button[ui.button_count].y = y;
    ui.button[ui.button_count].w = w;
    ui.button[ui.button_count].h = h;
    ui.button[ui.button_count].text[0] = 0;
    ui.button_count++;
  }
}

// plain text button
void drawButton( uint16_t x, uint16_t y, uint16_t w, uint16_t h, const char *text, uint8_t font_size, boolean active )
{
  tft.setFontAdafruit();
  tft.fillRoundRect( x,y,w,h,8, active ? BUTTON_HILITE:BUTTON_COLOR );
  tft.drawRoundRect( x,y,w,h,8,BUTTON_BORDER);
  tft.setTextSize(font_size);
  uint16_t tw = tft.measureTextWidth( text );
  uint16_t th = tft.measureTextHeight( "X" );
  tft.setCursor( x + (w-tw)/2, y + (h-th)/2 );
  tft.setTextColor(BUTTON_TEXT);
  tft.print(text);
  // add to button array
  addButton( x, y, w, h, text );
}

void drawButton( uint16_t x, uint16_t y, uint16_t w, uint16_t h, const char *text, boolean active )
{
  drawButton( x, y, w, h, text, 2, active );
}

void drawButton( uint16_t x, uint16_t y, uint16_t w, uint16_t h, ILI9341_t3_font_t f, const char *text )
{
  tft.fillRoundRect( x,y,w,h,8,BUTTON_COLOR );
  tft.drawRoundRect( x,y,w,h,8,BUTTON_BORDER);
  tft.setFont(f);
  uint16_t tw = tft.measureTextWidth( text );
  uint16_t th = tft.measureTextHeight( "X" );
  tft.setCursor( x + (w-tw)/2, y + (h-th)/2 );
  tft.setTextColor(BUTTON_TEXT);
  tft.print(text);
  // add to button array
  addButton( x, y, w, h, text );
}

// replot button from memory
void drawButton( uint8_t button_number, boolean active )
{
  if (button_number >= ui.button_count) return;
  drawButton( ui.button[button_number-1].x, ui.button[button_number-1].y, 
              ui.button[button_number-1].w, ui.button[button_number-1].h,
              ui.button[button_number-1].text, active );
}    

// visual feedback for button hit
void illuminate_button( uint8_t button_number )
{
  if (button_number >= ui.button_count) return;
  drawButton( button_number, true );
  delay(100);
  drawButton( button_number, false );
}

void drawCheckbox( uint16_t x, uint16_t y, uint16_t w, uint16_t h, boolean checked, ILI9341_t3_font_t f, const char *text )
{
  const int WID = 20; // check box dimensions
  const int HGT = 20;
  const int MARGIN = 40;
  tft.fillRoundRect( x,y,w,h,8,BUTTON_COLOR );
  tft.drawRoundRect( x,y,w,h,8,BUTTON_BORDER);
  tft.drawRect( x+(MARGIN-WID)/2, y+(h-HGT)/2, WID,HGT, BUTTON_TEXT);
  if (checked)
  { tft.drawLine( x+(MARGIN-WID)/2, y+(h-HGT)/2, x+(MARGIN-WID)/2+WID, y+(h-HGT)/2+HGT, BUTTON_TEXT);
    tft.drawLine( x+(MARGIN-WID)/2, y+(h-HGT)/2+HGT, x+(MARGIN-WID)/2+WID, y+(h-HGT)/2, BUTTON_TEXT);
  }
  tft.setFont(f);
  uint16_t th = tft.measureTextHeight( "X" );
  tft.setCursor( x + MARGIN, y + (h-th)/2 );
  tft.setTextColor(BUTTON_TEXT);
  tft.print(text);
  // add to button array
  addButton( x, y, w, h, text );
}

// display arbitrary text
void drawText( uint16_t x, uint16_t y, uint16_t text_color, uint8_t text_size, const char *text )
{
  tft.setCursor( x,y );
  tft.setTextColor( text_color );
  tft.setTextSize( text_size );
  tft.print( text );
}

// display centered text
void drawCenteredText( uint16_t y, uint16_t text_color, uint8_t text_size, const char *text )
{
  tft.setTextSize( text_size );
  uint16_t x = (DISP_W-tft.measureTextWidth(text)) / 2;
  drawText( x,y,text_color,text_size,text );
}
void drawCenteredText( uint16_t y, uint16_t text_color, const char *text )
{
  uint16_t x = (DISP_W-tft.measureTextWidth(text)) / 2;
  tft.setCursor( x,y );
  tft.setTextColor( text_color );
  tft.print( text );
}

//- - - - - - - - - - - - - - - - - - - - - -
 
#define CAL_BUTTON_SIZE 50
#define CAL_OFFSET (CAL_BUTTON_SIZE/2)

void displayCalibrationScreen()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setFontAdafruit();
  drawText( 20, 80, SETTINGS_TEXT_COLOR, 3, "Calibration" );
  drawText( 27,110, SETTINGS_TEXT_COLOR, 1, "Touch the calibration points" );
  // upper left corner (0,0)
  ui.button_count = 0;
  drawButton( 0,0, CAL_BUTTON_SIZE,CAL_BUTTON_SIZE, "1", false );
}

boolean handleCalibrationTouch( TS_Point p ) // returns true to end calibration
{
  static uint16_t cal_point[3][2]; // 1'st 3 calibration points

  switch (ui.mode)
  {
    case UI_MODE_CAL_POINT+0:
      if (p.x > 1500 || p.y > 2000) break; // sanity check    
      illuminate_button( 1 );
      cal_point[0][0] = p.x;
      cal_point[0][1] = p.y;  
      ui.mode++;
      // upper right corner
      tft.fillRect( 0,0, CAL_BUTTON_SIZE,CAL_BUTTON_SIZE, BACKGND_COLOR );
      ui.button_count = 0; // prepare to show next button
      drawButton( DISP_W-CAL_BUTTON_SIZE,0, CAL_BUTTON_SIZE,CAL_BUTTON_SIZE, "2", false );
      break;    
    case UI_MODE_CAL_POINT+1:
      if (p.x < 1500 || p.y > 2000) break; // sanity check    
      illuminate_button( 1 );
      cal_point[1][0] = p.x;
      cal_point[1][1] = p.y;  
      ui.mode++;
      // lower right corner
      tft.fillRect( DISP_W-CAL_BUTTON_SIZE,0, CAL_BUTTON_SIZE,CAL_BUTTON_SIZE, BACKGND_COLOR );
      ui.button_count = 0; // prepare to show next button
      drawButton( DISP_W-CAL_BUTTON_SIZE,DISP_H-CAL_BUTTON_SIZE, CAL_BUTTON_SIZE,CAL_BUTTON_SIZE, "3", false );
      break;    
    case UI_MODE_CAL_POINT+2:
      if (p.x < 1500 || p.y < 2000) break; // sanity check    
      illuminate_button( 1 );
      cal_point[2][0] = p.x;
      cal_point[2][1] = p.y;  
      ui.mode++;
      // lower left corner
      tft.fillRect( DISP_W-CAL_BUTTON_SIZE,DISP_H-CAL_BUTTON_SIZE, CAL_BUTTON_SIZE,CAL_BUTTON_SIZE, BACKGND_COLOR );
      ui.button_count = 0; // prepare to show next button
      drawButton( 0,DISP_H-CAL_BUTTON_SIZE, CAL_BUTTON_SIZE,CAL_BUTTON_SIZE, "4", false );
      break;    
    case UI_MODE_CAL_POINT+3: // last calibration point
    { illuminate_button( 1 );
      if (p.x > 1500 || p.y < 2000) break; // sanity check    
      int16_t vmin,vmax;
      vmin = (cal_point[0][0] + p.x) / 2;
      vmax = (cal_point[1][0] + cal_point[2][0]) / 2;
      ui_config.touch_cal.min_x = map(   0, CAL_OFFSET,DISP_W-CAL_OFFSET, vmin,vmax );
      ui_config.touch_cal.max_x = map( DISP_W, CAL_OFFSET,DISP_W-CAL_OFFSET, vmin,vmax );
      vmin = (cal_point[0][1] + cal_point[1][1]) / 2;
      vmax = (cal_point[2][1] + p.y) / 2;
      ui_config.touch_cal.min_y = map(      0, CAL_OFFSET,DISP_H-CAL_OFFSET, vmin,vmax );
      ui_config.touch_cal.max_y = map( DISP_H, CAL_OFFSET,DISP_H-CAL_OFFSET, vmin,vmax );
      // show confirmation screen
      tft.fillRect( 0,DISP_H-CAL_BUTTON_SIZE, CAL_BUTTON_SIZE,CAL_BUTTON_SIZE, BACKGND_COLOR );
      ui.mode++;
      ui.button_count = 0;
      tft.fillRect( 25,110, DISP_W-25,DISP_H-110, BACKGND_COLOR);
      drawText( 45,115, SETTINGS_TEXT_COLOR, 1, "Test Touch Calibration" );
      drawText( 35,130, SETTINGS_TEXT_COLOR, 1, "Tap OK or wait for timeout" );
      drawButton( DISP_W/2-50/2,190, 50,50, "OK", false );
      break;
    }
    case 104: // confirmation
    {
      // scale using calibration
      p.x = map(p.x, ui_config.touch_cal.min_x, ui_config.touch_cal.max_x, 0, tft.width() );
      p.y = map(p.y, ui_config.touch_cal.min_y, ui_config.touch_cal.max_y, 0, tft.height());
      // hilite point touched
      tft.fillCircle( p.x,p.y, 25, ILI9341_RED );
      delay(500);
      tft.fillCircle( p.x,p.y, 25, BACKGND_COLOR );
      // search button table
      for (uint8_t i=0; i < ui.button_count; i++)
        if (p.x >= ui.button[i].x && p.x <= ui.button[i].x+ui.button[i].w
         && p.y >= ui.button[i].y && p.y <= ui.button[i].y+ui.button[i].h)
         { illuminate_button( 1 );
           ui.update_pending = true; // |= UI_UPDATE_FLAG_CONFIG;
           ui.update_tm = millis();
           return true;
         }
    }    
  }
  return false;
}

//- - - - - - - - - - - - - - - - - - - - - -
// Main screen

#define SET_COLOR    ILI9341_CYAN
#define ACTUAL_COLOR ILI9341_GREEN
#define OFF_COLOR    ILI9341_RED
#define ALT_COLOR    ILI9341_WHITE

#define MAIN_L_MARGIN 50
#define MAIN_TOP_ACT  25
#define MAIN_TOP_SET 135

void paintCCCV( boolean redraw )
{
  if (ui.mode != UI_MODE_MAIN) return;
  tft.setFont(Arial_10_Bold);
  tft.setTextColor( BACKGND_COLOR );
  if (ui.cccv == 1)
  { tft.setCursor( MAIN_L_MARGIN-35,MAIN_TOP_ACT+17 );
    tft.fillRect( MAIN_L_MARGIN-35,MAIN_TOP_ACT+15, 29,15, ACTUAL_COLOR );
    tft.print( " CV " ); 
  }
  if (ui.cccv == -1)
  { tft.setCursor( MAIN_L_MARGIN-35,MAIN_TOP_ACT+65 );
    tft.fillRect( MAIN_L_MARGIN-35,MAIN_TOP_ACT+63, 29,15, ACTUAL_COLOR );
    tft.print( " CC " ); 
  }
  if (redraw)
  { if (ui.cccv != -1) tft.fillRect( MAIN_L_MARGIN-35,MAIN_TOP_ACT+63, 29,15, BACKGND_COLOR );
    if (ui.cccv !=  1) tft.fillRect( MAIN_L_MARGIN-35,MAIN_TOP_ACT+15, 29,15, BACKGND_COLOR );
  }
}

void paintActualV( boolean redraw )
{
  if (ui.mode != UI_MODE_MAIN) return;
  tft.setFont(Arial_40_Bold);
  tft.setTextColor( psu.is_on ? ACTUAL_COLOR : OFF_COLOR );
  char buf[6];
  dtostrf( ui.vs/100.0, 5,2, buf );
  uint16_t tw0  = tft.measureTextWidth( "0" );
  uint16_t twSP = tft.measureTextWidth( " " );
  tft.setCursor( MAIN_L_MARGIN + (buf[0]==' ' ? (tw0-twSP):0),MAIN_TOP_ACT );
  if (redraw) tft.fillRect( MAIN_L_MARGIN,MAIN_TOP_ACT, 140,41, BACKGND_COLOR );
  tft.print( buf );
  if (!redraw) tft.print("V"); 
}

void paintActualI( boolean redraw )
{
  if (ui.mode != UI_MODE_MAIN) return;
  tft.setFont(Arial_40_Bold);
  tft.setTextColor(  psu.is_on ? ACTUAL_COLOR : OFF_COLOR );
  uint16_t tw0  = tft.measureTextWidth( "0" );
  uint16_t twSP = tft.measureTextWidth( " " );
  if (ui.whcs)
  { uint16_t w_x10 = ((unsigned long)ui.vs * (unsigned long)ui.cs) / 1000UL;
    char buf[6];
    dtostrf( w_x10/10.0, 5,1, buf );
    tft.setCursor( MAIN_L_MARGIN + (buf[0]==' ' ? (tw0-twSP):0) + (buf[1]==' ' ? (tw0-twSP):0),MAIN_TOP_ACT+50 );
    if (redraw) tft.fillRect( MAIN_L_MARGIN,MAIN_TOP_ACT+50, 140,41, BACKGND_COLOR );
    tft.print( buf );
    if (!redraw) { tft.setFont(Arial_32_Bold);  tft.setCursor( MAIN_L_MARGIN + 140, MAIN_TOP_ACT+50+6 );  tft.print("W"); }
  }
  else
  { char buf[6];
    dtostrf( ui.cs/100.0, 5,2, buf );
    tft.setCursor( MAIN_L_MARGIN + (buf[0]==' ' ? (tw0-twSP):0),MAIN_TOP_ACT+50 );
    if (redraw) tft.fillRect( MAIN_L_MARGIN,MAIN_TOP_ACT+50, 140,41, BACKGND_COLOR );
    tft.print( buf );
    if (!redraw) tft.print("A"); 
  }
}

void paintSetV( boolean redraw )
{
  ui.cv_display_pend = false;
  if (ui.mode != UI_MODE_MAIN) return;
  tft.setTextColor( ui.v_altcolor ? ALT_COLOR:SET_COLOR );
  tft.setFont(Arial_40_Bold);
  char buf[6];
  dtostrf( ui_settings.cv/100.0, 5,2, buf );
  uint16_t tw0  = tft.measureTextWidth( "0" );
  uint16_t twSP = tft.measureTextWidth( " " );
  tft.setCursor( MAIN_L_MARGIN + (buf[0]==' ' ? (tw0-twSP):0),MAIN_TOP_SET );
  if (redraw) tft.fillRect( MAIN_L_MARGIN,MAIN_TOP_SET, 140,41, BACKGND_COLOR );
  tft.print( buf );
  if (!redraw) { tft.setTextColor( SET_COLOR );  tft.print("V"); }
}

void paintSetI( boolean redraw )
{      
  ui.cc_display_pend = false;
  if (ui.mode != UI_MODE_MAIN) return;
  tft.setTextColor( ui.c_altcolor ? ALT_COLOR:SET_COLOR );
  tft.setFont(Arial_40_Bold);
  char buf[6];
  dtostrf( ui_settings.cc/100.0, 5,2, buf );
  uint16_t tw0  = tft.measureTextWidth( "0" );
  uint16_t twSP = tft.measureTextWidth( " " );
  tft.setCursor( MAIN_L_MARGIN + (buf[0]==' ' ? (tw0-twSP):0),MAIN_TOP_SET+50 );
  if (redraw) tft.fillRect( MAIN_L_MARGIN,MAIN_TOP_SET+50, 140,41, BACKGND_COLOR );
  tft.print( buf );
  if (!redraw) { tft.setTextColor( SET_COLOR );  tft.print("A"); }
}

void paintSetIcon( boolean redraw )
{
  if (redraw)
  { tft.fillRect( 15,MAIN_TOP_SET+ 5, 30,30, BACKGND_COLOR );
    tft.fillRect( 15,MAIN_TOP_SET+55, 30,30, BACKGND_COLOR );
  }
  if (!ui.setting_cc)
       tft.writeRect( 15,MAIN_TOP_SET+ 5, 30,30, (uint16_t*)knob_icon);
  else tft.writeRect( 15,MAIN_TOP_SET+55, 30,30, (uint16_t*)knob_icon);
}

void paintOutputIndicator( boolean redraw )
{
  tft.setFont(Arial_10_Bold);
  tft.setTextColor( ACTUAL_COLOR );
  tft.setCursor( 5,1 );
  if (!redraw)
    tft.print( "Output ");
  else
  { tft.fillRect( 58,1, 28,10, BACKGND_COLOR );
    tft.setCursor( 58, 1 );
  }
  tft.print( psu.is_on ? "ON":"OFF" );
}

void paintLimitIndicator()
{
  if (ui_config.safe_limit)
  { tft.setFont(Arial_10_Bold);
    tft.setCursor( 100, 1 );
    tft.setTextColor( LIMIT_COLOR );
    tft.print( SafeLimitText(ui_config.safe_limit) );
    tft.print( "V Safe" );
  }
}

void paintTempIndicator( boolean redraw )
{
  if (ui.mode != UI_MODE_MAIN) return;
  tft.setFont(Arial_10_Bold);
  tft.setTextColor( TEMP_COLOR );
  if (!redraw)
  { tft.setCursor( 170, 1 );
    tft.print( "Temp" ); 
  }
  else
    tft.fillRect( 210,1, 28,10, BACKGND_COLOR );
  
  tft.setCursor( 210, 1 );
  tft.print( ui.temp ); 
  tft.print( "C" ); 
}


void displayMainScreen()
{
  const int BTN_HGT = 60;
  const int BTN_WID = 60;

  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);

  paintOutputIndicator( false );
  paintLimitIndicator( );
  paintTempIndicator( false );
  
  tft.setTextColor( ACTUAL_COLOR );
  tft.setFont(Arial_8_Bold);
  tft.setCursor( MAIN_L_MARGIN-40,MAIN_TOP_ACT+39 );
  tft.print( "Output" ); 

  paintActualV( false );
  addButton( 0,MAIN_TOP_ACT, DISP_W,BTN_HGT );  // button #1
  paintActualI( false );
  addButton( 0,MAIN_TOP_ACT+50, DISP_W,BTN_HGT );  // button #2

  paintCCCV( false );
  
  tft.setTextColor( SET_COLOR );
  tft.setFont(Arial_8_Bold);
  tft.setCursor( MAIN_L_MARGIN-30,MAIN_TOP_SET+40 );
  tft.print( "Set" ); 
  paintSetV( false );
  addButton( 0,MAIN_TOP_SET,    DISP_W,BTN_HGT );  // button #3
  paintSetI( false );
  addButton( 0,MAIN_TOP_SET+50, DISP_W,BTN_HGT );  // button #4

  paintSetIcon( false );

  tft.setFont(Arial_10_Bold);
  tft.setTextColor( TEXT_COLOR );
  tft.setCursor( 10,DISP_H-78 );
  tft.print( "On/Off" ); 
  tft.setCursor( DISP_W/2-BTN_WID/2+2,DISP_H-78 );
  tft.print( "Preset" ); 
  tft.setCursor( DISP_W-BTN_WID-5,DISP_H-78 );
  tft.print( "Settings" ); 

  tft.writeRect( 5,DISP_H-65, BTN_WID,BTN_HGT, (uint16_t*)power60);
  addButton( 5,DISP_H-65, BTN_WID,BTN_HGT );                          // button #5

  tft.writeRect( DISP_W/2-BTN_WID/2,DISP_H-BTN_HGT-5, BTN_WID,BTN_HGT, (uint16_t*)presets60);
  addButton( DISP_W/2-BTN_WID/2,DISP_H-BTN_HGT-5, BTN_WID,BTN_WID );  // button #6

  tft.writeRect( DISP_W-BTN_WID-5,  DISP_H-BTN_HGT-5, BTN_WID,BTN_HGT, (uint16_t*)wrench60);
  addButton( DISP_W-BTN_WID-5,  DISP_H-BTN_HGT-5, BTN_WID,BTN_HGT );  // button #7
}    

void handleMainTouch( uint8_t button_number )
{
  switch (button_number)
  {
    case 1: // VS
      ui.mode = UI_MODE_LIMITS;
      displayLimitsScreen();
      break;
      
    case 2: // CS
      ui.whcs = !ui.whcs; // toggle A/W display mode
      tft.fillRect( MAIN_L_MARGIN,MAIN_TOP_ACT+50, DISP_W-MAIN_L_MARGIN,41, BACKGND_COLOR );
      paintActualI( false );
      break;

    case 3: // CV
      ui.setting_cc = false;
      paintSetIcon( true );
      break;
      
    case 4: // CC
      ui.setting_cc = true;
      paintSetIcon( true );
      break;
    
    case 5: // on/off
      if (psu.is_on) // turn it off
      { stopPSU(); 
        paintOutputIndicator( true );
        paintActualV( true ); paintActualV( false );
        paintActualI( true ); paintActualI( false );
      }
      else // open power control screen
      { ui.mode = UI_MODE_POWER;
        displayPowerScreen();
      }
      break;

    case 6: // presets
      ui.mode = UI_MODE_PRESETS;
      displayPresetsScreen();   
      break;
    
    case 7: // settings
      ui.mode = UI_MODE_SETTINGS;
      displaySettingsScreen();
  }
}

void handleMainButton( boolean short_press )
{
  if (short_press) // toggle V/A jog
  {   ui.setting_cc = !ui.setting_cc;
      paintSetIcon( true );
  }
  else // long press toggles PSU on/off
  { if (psu.is_on)
         stopPSU();
    else startPSU();
    paintOutputIndicator( true );
    paintActualV( true ); paintActualV( false );
    paintActualI( true ); paintActualI( false );
  }
}

void handleMainDial( int rot )
{
  if (!ui.setting_cc)
  { ui_settings.cv = constrain( ui_settings.cv+rot, 0, getLimitVx100( ui_config.safe_limit ) );
    ui.cv_display_pend = true;
    setCV( ui_settings.cv );
    ui.v_altcolor = true;
    ui.update_pending = true; // |= UI_UPDATE_FLAG_SETTING;
    ui.update_tm = millis();
  }
  else
  { ui_settings.cc = constrain( ui_settings.cc+rot, 0, psu_specs[ ui_config.model ].max_a_x100 );
    ui.cc_display_pend = true;
    setCC( ui_settings.cc );
    ui.c_altcolor = true;
    ui.update_pending = true; // |= UI_UPDATE_FLAG_SETTING;
    ui.update_tm = millis();
  }
}

//- - - - - - - - - - - - - - - - - - - - - -
// Power Control screen
  
void displayPowerScreen()
{
  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText( 10, TEXT_COLOR, "Output Control" ); 
  tft.setFont(Arial_18_Bold);
  tft.setCursor( 25,40 );
  tft.print( ui_settings.cv/100.0, 2 ); tft.print("V"); 
  tft.setCursor( 125,40 );
  tft.print( ui_settings.cc/100.0, 2 ); tft.print("A");
  drawButton( DISP_W/2-150/2, 80, 150,100, Arial_20_Bold, "Output ON" ); // button #1
  drawButton( DISP_W/2-150/2,200, 150,100, Arial_20_Bold, "Cancel" );    // button #2
}

void handlePowerTouch( uint8_t button_number )
{
  switch (button_number)
  { case 1: // ok
      startPSU();
    case 2: // cancel
      ui.mode = UI_MODE_MAIN;
      displayMainScreen();
  }
}

void handlePowerButton()
{
  handlePowerTouch( 1 ); // same as Ok
}

//- - - - - - - - - - - - - - - - - - - - - -
// Settings screen

void displaySettingsScreen()
{
  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText( 5, TEXT_COLOR, "Settings" );
  const int WID = 175; 
  const int HGT =  50;
  const int BASE = 30;
  const int SP   = 55;
  drawButton( DISP_W/2-WID/2,BASE,      WID,HGT, Arial_14_Bold, "Brightness" );         // button #1
  drawButton( DISP_W/2-WID/2,BASE+SP*1, WID,HGT, Arial_14_Bold, "Safe Limit" );         // button #2
  drawButton( DISP_W/2-WID/2,BASE+SP*2, WID,HGT, Arial_14_Bold, "PSU Calibration" );    // button #3
  drawButton( DISP_W/2-WID/2,BASE+SP*3, WID,HGT, Arial_14_Bold, "Touch Calibration" );  // button #4
  drawButton( DISP_W/2-WID/2,BASE+SP*4, WID,HGT, Arial_14_Bold, "Exit" );               // button #5
}

void handleSettingsTouch( uint8_t button_number )
{
  switch (button_number)
  { case 1: // brightness
      ui.mode = UI_MODE_BRIGHTNESS;
      displayBrightnessScreen();
      break;
      
    case 2: // safe limits
      ui.mode = UI_MODE_LIMITS;
      displayLimitsScreen();
      break;
    
    case 3: // PSU cal
      ui.mode = UI_MODE_MODEL;     // start with model identification screen
      displayPSUmodelScreen();
      break;
    
    case 4: // touch cal
      ui.mode = UI_MODE_CAL_START; // start touch calibration
      break;  

    case 5: // exit
      ui.mode = UI_MODE_MAIN;
      displayMainScreen();
  }
}

void handleSettingsButton()
{
  ui.mode = UI_MODE_INFO;
  displayInfoScreen();
}

//- - - - - - - - - - - - - - - - - - - - - -
// Brightness screen

void paintBrightness()
{
  tft.fillRect( 100,42, 50,50, BACKGND_COLOR );
  tft.setTextColor( TEXT_COLOR );
  tft.setFont(Arial_32_Bold);
  tft.setCursor( 100,42 );
  tft.print( ui_config.brightness+1 ); 
}

void displayBrightnessScreen()
{
  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText( 5, TEXT_COLOR, "Brightness" );
  paintBrightness();  
  const int WID = 175; 
  const int HGT =  60;
  const int BASE = 100;
  const int SP = 80;
  drawButton( DISP_W/2-WID/2,BASE+SP*0, WID,HGT, Arial_14_Bold, "Ok" );     // button #1
  drawButton( DISP_W/2-WID/2,BASE+SP*1, WID,HGT, Arial_14_Bold, "Cancel" ); // button #2
}

void handleBrightnessTouch( uint8_t button_number )
{
  switch (button_number)
  { case 1: // ok
      ui.update_pending = true; //  |= UI_UPDATE_FLAG_CONFIG;
      ui.update_tm = millis();
      break;
    case 2: // cancel
      ui_config.brightness = ee_config.brightness; // restore original setting
      setBrightness( ui_config.brightness );
  }
  ui.mode = UI_MODE_MAIN;
  displayMainScreen();
}

void handleBrightnessButton()
{
  handleBrightnessTouch( 1 ); // same as Ok
}

void handleBrightnessDial( int rot )
{
  ui_config.brightness = constrain( ui_config.brightness + rot, 0, 9 );
  paintBrightness();
  setBrightness( ui_config.brightness );
}

//- - - - - - - - - - - - - - - - - - - - - -
// Safe Limits screen

void displayLimitsScreen()
{
  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText( 5, TEXT_COLOR, "Safe Limit" );
  const int WID = 175; 
  const int HGT =  50;
  const int BASE = 30;
  const int SP   = 55;
  drawCheckbox( DISP_W/2-WID/2,BASE+SP*0, WID,HGT, (ui_config.safe_limit == 1), Arial_14_Bold, psu_specs[ui_config.model].limit_label[0] ); // button #1
  drawCheckbox( DISP_W/2-WID/2,BASE+SP*1, WID,HGT, (ui_config.safe_limit == 2), Arial_14_Bold, psu_specs[ui_config.model].limit_label[1] ); // button #2
  drawCheckbox( DISP_W/2-WID/2,BASE+SP*2, WID,HGT, (ui_config.safe_limit == 3), Arial_14_Bold, psu_specs[ui_config.model].limit_label[2] ); // button #3
  drawCheckbox( DISP_W/2-WID/2,BASE+SP*3, WID,HGT, (ui_config.safe_limit == 0), Arial_14_Bold, "No Limit" );                                // button #4
  drawButton( DISP_W/2-WID/2,BASE+SP*4, WID,HGT, Arial_14_Bold, "Cancel" );                                                                 // button #5
}

void handleLimitsTouch( uint8_t button_number )
{
  switch (button_number)
  { case 1: ui_config.safe_limit = 1; break;
    case 2: ui_config.safe_limit = 2; break;
    case 3: ui_config.safe_limit = 3; break;
    case 4: ui_config.safe_limit = 0; break;
  }
  ui_settings.cv = constrain( ui_settings.cv, 0, getLimitVx100( ui_config.safe_limit ));
  setCV(ui_settings.cv);
  ui.update_pending = true; // |= UI_UPDATE_FLAG_CONFIG;
  ui.update_tm = millis();
  ui.mode = UI_MODE_MAIN;
  displayMainScreen();
}

void handleLimitsButton()
{
  handleLimitsTouch( 5 ); // same as Cancel
}

//- - - - - - - - - - - - - - - - - - - - - -
// Presets screen

void displayPresetsScreen()
{
  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText( 5, TEXT_COLOR, "Presets" );
  const int COL1 =  30;
  const int COL2 = 130;
  const int WID  =  80;
  const int CAN_WID = 175; 
  const int HGT =  50;
  const int BASE = 30;
  const int SP   = 55;
  // voltage column
  drawButton( COL1,BASE+SP*0, WID,HGT, Arial_14_Bold, psu_specs[ ui_config.model ].preset_v_label[0] );  // button #1
  drawButton( COL1,BASE+SP*1, WID,HGT, Arial_14_Bold, psu_specs[ ui_config.model ].preset_v_label[1] );  // button #2
  drawButton( COL1,BASE+SP*2, WID,HGT, Arial_14_Bold, psu_specs[ ui_config.model ].preset_v_label[2] );  // button #3
  drawButton( COL1,BASE+SP*3, WID,HGT, Arial_14_Bold, psu_specs[ ui_config.model ].preset_v_label[3] );  // button #4
  // current column
  drawButton( COL2,BASE+SP*0, WID,HGT, Arial_14_Bold, psu_specs[ ui_config.model ].preset_a_label[0] );  // button #5
  drawButton( COL2,BASE+SP*1, WID,HGT, Arial_14_Bold, psu_specs[ ui_config.model ].preset_a_label[1] );  // button #6
  drawButton( COL2,BASE+SP*2, WID,HGT, Arial_14_Bold, psu_specs[ ui_config.model ].preset_a_label[2] );  // button #7
  drawButton( COL2,BASE+SP*3, WID,HGT, Arial_14_Bold, psu_specs[ ui_config.model ].preset_a_label[3] );  // button #8

  drawButton( DISP_W/2-CAN_WID/2,BASE+SP*4, CAN_WID,HGT, Arial_14_Bold, "Cancel" );                      // button #9
}

void handlePresetsTouch( uint8_t button_number )
{
  uint16_t ncv = 0;
  uint16_t ncc = 0;
  switch (button_number)
  { case 1: ncv = psu_specs[ ui_config.model ].preset_v[ 0 ];  break;
    case 2: ncv = psu_specs[ ui_config.model ].preset_v[ 1 ];  break;
    case 3: ncv = psu_specs[ ui_config.model ].preset_v[ 2 ];  break;
    case 4: ncv = psu_specs[ ui_config.model ].preset_v[ 3 ];  break;
    case 5: ncc = psu_specs[ ui_config.model ].preset_a[ 0 ];  break;
    case 6: ncc = psu_specs[ ui_config.model ].preset_a[ 1 ];  break;
    case 7: ncc = psu_specs[ ui_config.model ].preset_a[ 2 ];  break;
    case 8: ncc = psu_specs[ ui_config.model ].preset_a[ 3 ];  break;
  }

  if (ncv)
  { ui_settings.cv = constrain( ncv, 0, getLimitVx100( ui_config.safe_limit ));
    setCV(ui_settings.cv);
  }
  if (ncc)
  { ui_settings.cc = ncc;
    setCC(ui_settings.cc);
  }
  if (ncc | ncv)
  { ui.update_pending = true; //  |= UI_UPDATE_FLAG_SETTING;
    ui.update_tm = millis();
  }
  ui.mode = UI_MODE_MAIN;
  displayMainScreen();
}

void handlePresetsButton()
{
  handlePresetsTouch( 9 ); // same as Cancel
}

//- - - - - - - - - - - - - - - - - - - - - -

void displayInfoScreen()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText( 10, TEXT_COLOR, "System Information" );

  tft.setFontAdafruit();
  const int PS = 25;
  const int LS = 11;

  int Y = 35;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "PSU Model: " );
  tft.print( psu_specs[ui_config.model].name );
  Y += 13;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "UI Version: 2020.09.16" );
  Y += 13;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "Safe Voltage Limit: " );
  tft.print( SafeLimitText( ui_config.safe_limit ) );
  if (ui_config.safe_limit) tft.print( "V" );
  Y += 13;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "Dispaly Brightness: " );
  tft.print( String( ui_config.brightness+1 ) );
  Y += 13;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "EEPROM Settings Blocks: " );
  tft.print( String( ui_config.num_blocks ) );
  Y += 13;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "Current Settings Block: " );
  tft.print( String( ee_current_settings_block ) );
  Y += 13;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "Settings Writes Used: " );
  tft.print( String( ee_write_count ) );
  Y += 13;
  if (ee_current_settings_block == -1)
    drawText( 2,Y, INFO_TEXT_COLOR, 1, "SETTINGS WRITES EXHAUSTED" );
  else
  { drawText( 2,Y, INFO_TEXT_COLOR, 1, "Settings Writes Left: " );
    tft.print( String( 100000UL * ui_config.num_blocks - ee_write_count ) );
  }  

  Y += 13;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "CV calibration:" );
  tft.setCursor( 10,Y+LS );
  for (int i=0; i < 6; i++)
  { if (i > 0) tft.print( ", " );
    tft.print( String( ui_config.cv_cal[i] ) );  
  }

  Y += PS;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "CC calibration:" );
  tft.setCursor( 10,Y+LS );
  for (int i=0; i < 4; i++)
  { if (i > 0) tft.print( ", " );
    tft.print( String( ui_config.cc_cal[i] ) );  
  }

  Y += PS;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "VS calibration:" );
  tft.setCursor( 10,Y+LS );
  for (int i=0; i < 6; i++)
  { if (i > 0) tft.print( ", " );
    tft.print( String( ui_config.vs_cal[i] ) );  
  }

  Y += PS;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "CS calibration:" );
  tft.setCursor( 10,Y+LS );
  for (int i=0; i < 4; i++)
  { if (i > 0) tft.print( ", " );
    tft.print( String( ui_config.cs_cal[i] ) );  
  }

  Y += PS;
  drawText( 2,Y, INFO_TEXT_COLOR, 1, "Touch calibration:" );
  tft.setCursor( 10,Y+11 );
  tft.print( String( ui_config.touch_cal.min_x ) );  tft.print( ", " );
  tft.print( String( ui_config.touch_cal.min_y ) );  tft.print( " - " );
  tft.print( String( ui_config.touch_cal.max_x ) );  tft.print( ", " );
  tft.print( String( ui_config.touch_cal.max_y ) );

  ui.button_count = 0;
  drawButton( 80,280, 90,40, Arial_14_Bold, "Ok" ); // button #1
}

void handleInfoInput() // touch or button
{
  ui.mode = UI_MODE_MAIN;
  displayMainScreen();
}

//- - - - - - - - - - - - - - - - - - - - - -
// PSU calibration : Model identification screen

void displayPSUmodelScreen()
{
  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText( 15, TEXT_COLOR, "PSU Model" );
  const int X   = 45;
  const int Y   = 50;
  const int WID =150;
  const int HGT = 50;
  const int SP  = 70;
  drawCheckbox( X,Y+SP*0, WID,HGT, (ui_config.model == 0), Arial_14_Bold, psu_specs[ 0 ].name ); // button #1
  drawCheckbox( X,Y+SP*1, WID,HGT, (ui_config.model == 1), Arial_14_Bold, psu_specs[ 1 ].name ); // button #2
  drawCheckbox( X,Y+SP*2, WID,HGT, (ui_config.model == 2), Arial_14_Bold, psu_specs[ 2 ].name ); // button #3
  drawButton(   X,Y+SP*3, WID,HGT,                         Arial_14_Bold, "Cancel" );            // button #4
}

void handlePSUmodelTouch( uint8_t button_number )
{
  switch (button_number)
  { case 1: 
    case 2:
    case 3:
      ui_config.model = button_number-1;
      // proceed to next screen (don't save changes until calibration is done)
      ui.mode = UI_MODE_PSU_CAL_V;
      displayPSUcalVScreen(0);
      break;
      
    case 4: // cancel
      ui_config = ee_config; // restore settings
      setCC( ui_settings.cc );
      setCV( ui_settings.cv );
      // exit to main screen
      ui.mode = UI_MODE_MAIN;
      displayMainScreen();
  }
}

//- - - - - - - - - - - - - - - - - - - - - -
// PSU calibration screen : Voltage calibration sequence

int      psu_cal_page;
uint16_t psu_cal_dac;

void paintPSUadc()
{
  tft.fillRect( 120,310, 100,10, BACKGND_COLOR );
  tft.setTextColor( TEXT_COLOR );
  tft.setFont(Arial_8_Bold);
  tft.setCursor( 120,310 );
  tft.print( psu_cal_dac ); 
}

void displayPSUcalVScreen( int page )
{
  // remember what page is current
  psu_cal_page = page;

  // get current cal setting for this point
  psu_cal_dac = ui_config.cv_cal[ page ]; 
  analogWriteResolution( 15 );
  analogWrite( PWM_CV_PIN, psu_cal_dac );
  
  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText(  5, TEXT_COLOR, "PSU Calibration" );
  drawCenteredText( 50, TEXT_COLOR, "Turn dial to" );
  drawCenteredText( 75, TEXT_COLOR, "output volage" );
  tft.setFont(Arial_32_Bold);
  tft.setCursor( 80,100 );
  tft.print( psu_specs[ ui_config.model ].v_cal_points[ page ] / 100.0, 2 ); 
  tft.print( "V" );
  const int WID = 80;
  const int HGT = 40;
  drawButton( 20,150, WID,HGT, Arial_14_Bold, "Ok" );     // button #1
  drawButton( 80,240, WID,HGT, Arial_14_Bold, "Cancel" ); // button #2
  drawButton(140,150, WID,HGT, Arial_14_Bold, "Back" );   // button #3
  paintPSUadc();
  
  // enable output
  startPSU();
  // set nominal current so we get some voltage
  analogWriteResolution( 15 );
  analogWrite( PWM_CC_PIN, 15000 );
}

void handlePSUcalVTouch( uint8_t button_number )
{
  switch (button_number)
  { case 1: // ok
      // store calibration point
      ui_config.cv_cal[ psu_cal_page ] = psu_cal_dac;
      ui_config.vs_cal[ psu_cal_page ] = psu.vs_adc;
      
      // proceed to next screen
      if (psu_cal_page >= 5)
      { stopPSU();
        analogWrite( PWM_CC_PIN, 0 );
        ui.mode = UI_MODE_PSU_CAL_SAVE1;
        displayPSUsave1Screen();
      }
      else
        displayPSUcalVScreen( psu_cal_page+1 );
      break; 

    case 2: // cancel
      ui_config = ee_config; // restore settings
      stopPSU();
      setCC( ui_settings.cc );
      setCV( ui_settings.cv );
      ui.mode = UI_MODE_MAIN;
      displayMainScreen();
      break;
      
    case 3: // back
      if (psu_cal_page)
        displayPSUcalVScreen( psu_cal_page-1 );
      else
      { stopPSU();
        ui.mode = UI_MODE_MODEL;
        displayPSUmodelScreen();
      }
  }
}

void handlePSUcalVButton()
{
  handlePSUcalVTouch( 1 ); // same as Ok
}

void handlePSUcalVDial( int rot )
{
  // sending DAC counts directly in this mode
  if (rot < 0 && psu_cal_dac < -rot) psu_cal_dac = 0;
  else psu_cal_dac += rot;
  if (psu_cal_dac > 0x7FFF) psu_cal_dac = 0x7FFF;
  analogWriteResolution( 15 );
  analogWrite( PWM_CV_PIN, psu_cal_dac );
  paintPSUadc();
}

//- - - - - - - - - - - - - - - - - - - - - -
// PSU calibration : First save screen (model & voltage cal)

void displayPSUsave1Screen()
{
  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText(  5, TEXT_COLOR, "PSU Calibration" );
  drawCenteredText( 50, TEXT_COLOR, "Ready to save the" );
  drawCenteredText( 75, TEXT_COLOR, "pump model and" );
  drawCenteredText(100, TEXT_COLOR, "voltage calibration." );
  const int WID = 80;
  const int HGT = 40;
  drawButton( 20,150, WID,HGT, Arial_14_Bold, "Ok" );     // button #1
  drawButton( 80,240, WID,HGT, Arial_14_Bold, "Cancel" ); // button #2
  drawButton(140,150, WID,HGT, Arial_14_Bold, "Back" );   // button #3
}

void handlePSUsave1Touch( uint8_t button_number )
{
  switch (button_number)
  { case 1: // ok
      saveChanges(); // copy ui_config -> ee_config -> EEPROM
      // proceed to next screen
      ui.mode = UI_MODE_PSU_CAL_A;
      displayPSUcalAScreen( 0 );
      break;
  
    case 2: // cancel
      ui_config = ee_config; // restore settings
      // exit to main screen
      stopPSU();
      setCC( ui_settings.cc );
      setCV( ui_settings.cv );
      ui.mode = UI_MODE_MAIN;
      displayMainScreen();
      break;
      
    case 3: // back
      ui.mode = UI_MODE_PSU_CAL_V;
      displayPSUcalVScreen( 5 );
  }
}

//- - - - - - - - - - - - - - - - - - - - - -
// PSU calibration screen : Current calibration sequence

void displayPSUcalAScreen( int page )
{
  // remember what page is current
  psu_cal_page = page;

  // get current cal setting for this point
  psu_cal_dac = ui_config.cc_cal[ page ]; 
  analogWriteResolution( 15 );
  analogWrite( PWM_CC_PIN, psu_cal_dac );

  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText(  5, TEXT_COLOR, "PSU Calibration" );
  drawCenteredText( 50, TEXT_COLOR, "Turn dial to" );
  drawCenteredText( 75, TEXT_COLOR, "output current" );
  tft.setFont(Arial_32_Bold);
  tft.setCursor( 80,100 );
  tft.print( psu_specs[ ui_config.model ].a_cal_points[ page ] / 100.0, 2 ); 
  tft.print( "A" );
  const int WID = 80;
  const int HGT = 40;
  drawButton( 20,150, WID,HGT, Arial_14_Bold, "Ok" );     // button #1
  drawButton( 80,240, WID,HGT, Arial_14_Bold, "Cancel" ); // button #2
  if (page)
  drawButton(140,150, WID,HGT, Arial_14_Bold, "Back" );   // button #3
  paintPSUadc();
  
  // set nominal voltage so we get some current
  startPSU();
  analogWriteResolution( 15 );
  analogWrite( PWM_CV_PIN, 5000 );
}

void handlePSUcalATouch( uint8_t button_number )
{
  switch (button_number)
  { case 1: // ok
      // store calibration point
      ui_config.cc_cal[ psu_cal_page ] = psu_cal_dac;
      ui_config.cs_cal[ psu_cal_page ] = psu.cs_adc;
      
      // proceed to next screen
      if (psu_cal_page >= 3)
      { ui.mode = UI_MODE_PSU_CAL_SAVE2;
        displayPSUsave2Screen();
      }
      else
        displayPSUcalAScreen( psu_cal_page+1 );
      break; 

    case 2: // cancel
      ui_config = ee_config; // restore settings
      stopPSU();
      setCC( ui_settings.cc );
      setCV( ui_settings.cv );
      ui.mode = UI_MODE_MAIN;
      displayMainScreen();
      break;
      
    case 3: // back
      displayPSUcalAScreen( psu_cal_page-1 );
  }
}

void handlePSUcalAButton()
{
  handlePSUcalATouch( 1 ); // same as Ok
}

void handlePSUcalADial( int rot )
{
  // sending DAC counts directly in this mode
  if (rot < 0 && psu_cal_dac < -rot) psu_cal_dac = 0;
  else psu_cal_dac += rot;
  if (psu_cal_dac > 0x7FFF) psu_cal_dac = 0x7FFF;
  analogWriteResolution( 15 );
  analogWrite( PWM_CC_PIN, psu_cal_dac );
  paintPSUadc();
}

//- - - - - - - - - - - - - - - - - - - - - -
// PSU calibration : Final save screen (current cal)

void displayPSUsave2Screen()
{
  ui.button_count = 0;
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText(  5, TEXT_COLOR, "PSU Calibration" );
  drawCenteredText( 50, TEXT_COLOR, "Ready to save the" );
  drawCenteredText( 75, TEXT_COLOR, "current calibration." );
  const int WID = 80;
  const int HGT = 40;
  drawButton( 20,150, WID,HGT, Arial_14_Bold, "Ok" );     // button #1
  drawButton( 80,240, WID,HGT, Arial_14_Bold, "Cancel" ); // button #2
  drawButton(140,150, WID,HGT, Arial_14_Bold, "Back" );   // button #3
}

void handlePSUsave2Touch( uint8_t button_number )
{
  switch (button_number)
  { case 1: // ok
      saveChanges(); // copy ui_config -> ee_config -> EEPROM
    case 2: // cancel
      ui_config = ee_config; // restore settings
      // exit to main screen
      stopPSU();
      setCC( ui_settings.cc );
      setCV( ui_settings.cv );
      ui.mode = UI_MODE_MAIN;
      displayMainScreen();
      break;
      
    case 3: // back
      ui.mode = UI_MODE_PSU_CAL_V;
      displayPSUcalAScreen( 4 );
  }
}

//- - - - - - - - - - - - - - - - - - - - - -
// Fatal Fault screen

void fault( const char *msg ) // never returns!
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setFont(Arial_14_Bold);
  drawCenteredText( 100, ERROR_TEXT_COLOR, "ERROR" );
  drawCenteredText( 140, ERROR_TEXT_COLOR, msg );
  while (1) ;
}

//-------------------------------------------------------------
// helper to figure determine if CC or CV mode

int calcCCCV()
{
  if (!psu.is_on) return 0;
  if (ui.cs >= 0.90 * ui.cc && ui.vs <= ui.cv) return -1;
  if (ui.vs >= 0.90 * ui.cv && ui.cs <= ui.cc) return  1;
  return 0; // dead zone
}

//-------------------------------------------------------------

void setup() 
{
  // diagnostics
  Serial.begin(115200); // debug monitor

  // set up digital pins
  digitalWrite( ENA_PUMP_PIN, LOW );  pinMode( ENA_PUMP_PIN, OUTPUT );
  pinMode( ENC_A_PIN,      INPUT_PULLUP );
  pinMode( ENC_B_PIN,      INPUT_PULLUP );
  pinMode( ENC_BUTTON_PIN, INPUT_PULLUP );

  // status LED
  status_pixel.begin();

  // CV & CC signals
  //   set pins 3 & 4 for 14 bit PWM
  analogWriteFrequency( 3, 1464.843 );

  // LCD
  //   3,1 puts UL corner near pins, in landscape mode
  //   0,2 puts UL away from pins, in portrait mode
  tft.begin();
  tft.setRotation(0);   // upper left corner is 0,0
  tft.fillScreen(BACKGND_COLOR);
  setBrightness( DEFAULT_BRIGHTNESS ); // default to a low but usable brightness setting
  // touch panel
  ts.begin();
  ts.setRotation(2);    // upper left corner is 0,0

  // start 1msec interrupt 
  myTimer.begin(int1msec, 1000);

  // get settings from EEPROM
  loadConfig();   // to ee_config
  loadSettings(); // to ee_settings
  // copy settings to ui
  ui_config   = ee_config;
  ui_settings = ee_settings;
  // set initial control voltages
  setCV( ui_settings.cv );
  setCC( ui_settings.cc );
  // set config brightness
  setBrightness( ui_config.brightness );

  // ADS1115 on I2C
  Wire.begin();
  if (!checkI2Cdevice( ADS1115_ADDR )) fault("ADC FAILED");

  // enter calibration mode if touched on powerup
  if (ts.touched() || first_time_setup)
  { 
    ui.mode = UI_MODE_CAL_START; // start calibration
  }
  else // show main screen
  {
    ui.mode = UI_MODE_MAIN;
    displayMainScreen();
  }
}

//-------------------------------------------------------------

void loop() 
{

//-- pump the power enable circuit

  if (psu.is_on)
    if (millis() - psu.pump_tm >= 5)
    { psu.pump_tm = millis();
      psu.pump_phase = !psu.pump_phase;
      digitalWrite( ENA_PUMP_PIN, psu.pump_phase );
    }

//-- watch for rotary encoder input

  if (encoder.value)
  { noInterrupts();
      int rot = encoder.value;
      encoder.value = 0;
    interrupts();
    switch (ui.mode)
    { case UI_MODE_MAIN:       handleMainDial(rot);           break;
      case UI_MODE_BRIGHTNESS: handleBrightnessDial(rot);     break;
      case UI_MODE_PSU_CAL_V:  handlePSUcalVDial(rot);        break;
      case UI_MODE_PSU_CAL_A:  handlePSUcalADial(rot);        break;
    }
  }

//-- watch for button input

  if (button.pressed)
  { button.pressed = false;
    switch (ui.mode)
    { case UI_MODE_MAIN:       handleMainButton(true);        break;
      case UI_MODE_POWER:      handlePowerButton();           break;
      case UI_MODE_SETTINGS:   handleSettingsButton();        break;
      case UI_MODE_BRIGHTNESS: handleBrightnessButton();      break;
      case UI_MODE_LIMITS:     handleLimitsButton();          break;
      case UI_MODE_PRESETS:    handlePresetsButton();         break;
      case UI_MODE_PSU_CAL_V:  handlePSUcalVButton();         break;
      case UI_MODE_PSU_CAL_A:  handlePSUcalAButton();         break;
      case UI_MODE_INFO:       handleInfoInput();             break;
    }
  }

  if (button.long_pressed)
  { button.long_pressed = false;
    switch (ui.mode)
    { case UI_MODE_MAIN:       handleMainButton(false);   break;
    }
  }

//-- watch for touch input

  { static boolean  was_touched;
    static uint32_t calibration_tm;
    boolean is_touched  = ts.touched();
    boolean touch_event = is_touched & !was_touched;

    if (ui.mode == UI_MODE_CAL_START) // starting calibration?
    {
      was_touched = true;
      ui.mode = UI_MODE_CAL_POINT; // 1'st calibration point
      displayCalibrationScreen();
      calibration_tm = millis();
    }
    else if (touch_event) // send touch events to settings handler
    {
      TS_Point p = ts.getPoint(); 
      if (ui.mode >= UI_MODE_CAL_POINT) // handle calibration mode
      {
        calibration_tm = millis(); // keep alive
        if (handleCalibrationTouch( p )) // using raw touch coordinates
        { if (first_time_setup)
          { first_time_setup = false;
            ui.mode = UI_MODE_MODEL;     // start PSU setup & calibration
            displayPSUmodelScreen();
          }
          else
          { ui.mode = UI_MODE_MAIN; // exit calibration mode upon completion
            displayMainScreen();
          }
        }
      }
      else // find touch button and send to current settings screen
      {      
        // scale using calibration
        p.x = map(p.x, ui_config.touch_cal.min_x, ui_config.touch_cal.max_x, 0, tft.width() );
        p.y = map(p.y, ui_config.touch_cal.min_y, ui_config.touch_cal.max_y, 0, tft.height());
        
        // search button table
        uint8_t button_index = 0;
        for (uint8_t i=0; i < ui.button_count; i++)
          if (p.x >= ui.button[i].x && p.x <= ui.button[i].x+ui.button[i].w
           && p.y >= ui.button[i].y && p.y <= ui.button[i].y+ui.button[i].h)
          { button_index = i+1;  break; }
  
        if (button_index) switch (ui.mode)
        { case UI_MODE_MAIN:          handleMainTouch      ( button_index );  break;
          case UI_MODE_POWER:         handlePowerTouch     ( button_index );  break;
          case UI_MODE_SETTINGS:      handleSettingsTouch  ( button_index );  break;
          case UI_MODE_BRIGHTNESS:    handleBrightnessTouch( button_index );  break;
          case UI_MODE_LIMITS:        handleLimitsTouch    ( button_index );  break;
          case UI_MODE_PRESETS:       handlePresetsTouch   ( button_index );  break;
          case UI_MODE_MODEL:         handlePSUmodelTouch  ( button_index );  break;
          case UI_MODE_PSU_CAL_V:     handlePSUcalVTouch   ( button_index );  break;
          case UI_MODE_PSU_CAL_SAVE1: handlePSUsave1Touch  ( button_index );  break;
          case UI_MODE_PSU_CAL_A:     handlePSUcalATouch   ( button_index );  break;
          case UI_MODE_PSU_CAL_SAVE2: handlePSUsave2Touch  ( button_index );  break;
          case UI_MODE_INFO:          handleInfoInput();                      break;
        }
      }
    }
    else if (ui.mode >= UI_MODE_CAL_POINT) // auto timeout calibration
    {
      if (millis() - calibration_tm > CAL_TIMEOUT_MS)
      {
        ui_config.touch_cal = ee_config.touch_cal; // restore calibration settings
        ui.mode = UI_MODE_MAIN;
        displayMainScreen();
      }
    }
  
    was_touched = is_touched;
  }


//-- status LED

  { int nstate;
    if (psu.is_on)
      nstate = 2;
    else if ((ui.vs >= 10) && ((ui.cv < 1000 && ui.vs >= ui.cv/10) || (ui.cv >= 1000 && ui.vs >= 100))) // 1V or 10%, whichever is less
      nstate = 1;
    else
      nstate = 0;

    if (nstate != status_led) paint_status_led( nstate );
  }
  
//-- read ADC

#define ADC_SAMPLE_RATE 60 // sample all channels every 240 msec

  { static unsigned long adc_tm   = millis();
    static int           adc_scan = 0;

    if (millis() - adc_tm >= ADC_SAMPLE_RATE)
    {
      adc_tm = millis();
  
      // adc full scale is 32767 @ 2.048V
      uint16_t adc = readRegister16(ADS1115_ADDR, ADS1015_REG_POINTER_CONVERT); 
      if (adc & 0x8000) adc = 0; // negative value
          
      switch (adc_scan)
      { case 0: // VS
        { // full scale is 32767 @ 2.048V
          psu.vs_adc = adc;
          uint8_t i;
          for (i=1; i < 5; i++) // locate calibration bracket
            if (adc <= ui_config.vs_cal[ i ]) break;
          int32_t nvs = map( adc, 
                             ui_config.vs_cal[ i-1 ], ui_config.vs_cal[ i ],
                             psu_specs[ ui_config.model ].v_cal_points[i-1], psu_specs[ ui_config.model ].v_cal_points[i] );
          if (nvs < 0) nvs = 0;                             
          // update display
          uint16_t nw  = ((unsigned long)nvs * (unsigned long)ui.cs) / 1000UL;
          boolean update_w = (nw != ((unsigned long)ui.vs * (unsigned long)ui.cs) / 1000UL);
          boolean update_v = (nvs != ui.vs);
          ui.vs = nvs;
          if (ui.mode == UI_MODE_MAIN) // only on main screen
          { if (update_v) paintActualV( true ); 
            if (update_w) paintActualI( true );
          }

          // voltage tracking
          if (calcCCCV() == 1) // only in CV mode
          {
            // calc error term, Vx100
            int16_t e = ui.cv - ui.vs;
          
            // disengage tracking if more than 10% out
            float epct = (float)e/(float)ui.vs;
              if (epct < 0) epct = -epct;
            if (epct <= MAX_TRACKING_CORRECTION)
            {
              // get error derivative, accumulate fractional remainder
              psu.cv_tracking_err_accumulator += e;
              e = psu.cv_tracking_err_accumulator / 2;
              psu.cv_tracking_err_accumulator -= e*2;
            
              // apply error term
              psu.cv_tracking_correction_pct += (float)e / (float)ui.vs;
              if (psu.cv_tracking_correction_pct < -MAX_TRACKING_CORRECTION) psu.cv_tracking_correction_pct = -MAX_TRACKING_CORRECTION;
              if (psu.cv_tracking_correction_pct >  MAX_TRACKING_CORRECTION) psu.cv_tracking_correction_pct =  MAX_TRACKING_CORRECTION;
            
              setCV(ui.cv);
            }
          }
        }
        break;
  
        case 1: // CS
        { // full scale is 32767 @ 2.048V
          psu.cs_adc = adc;
          uint8_t i;
          for (i=1; i < 3; i++) // locate calibration bracket
            if (adc <= ui_config.cs_cal[ i ]) break;
          int32_t a = map( adc, 
                           ui_config.cs_cal[ i-1 ], ui_config.cs_cal[ i ],
                           psu_specs[ ui_config.model ].a_cal_points[i-1], psu_specs[ ui_config.model ].a_cal_points[i] );
        if (a < 0) a = 0;                           
          // update display
          if (a != ui.cs)
          { ui.cs = a;
            if (ui.mode == UI_MODE_MAIN) // only on main screen
              paintActualI( true ); 
          }

          // current tracking
          if (calcCCCV() == -1) // only in CC mode
          {
            // calc error term, Ax100
            int16_t e = ui.cc - ui.cs;
          
            // disengage tracking if more than 10% out
            float epct = (float)e/(float)ui.cs;
              if (epct < 0) epct = -epct;
            if (epct <= MAX_TRACKING_CORRECTION)
            {
              // get error derivative, accumulate fractional remainder
              psu.cc_tracking_err_accumulator += e;
              e = psu.cc_tracking_err_accumulator / 2;
              psu.cc_tracking_err_accumulator -= e*2;
            
              // apply error term
              psu.cc_tracking_correction_pct += (float)e / (float)ui.cs;
              if (psu.cc_tracking_correction_pct < -MAX_TRACKING_CORRECTION) psu.cc_tracking_correction_pct = -MAX_TRACKING_CORRECTION;
              if (psu.cc_tracking_correction_pct >  MAX_TRACKING_CORRECTION) psu.cc_tracking_correction_pct =  MAX_TRACKING_CORRECTION;
            
              setCC(ui.cc);
            }
          }
        }
        break;
  
        case 2: // NTC
        { // 100K MuRata thermistor
          #define CtoK(C) (C+273.15)
          #define KtoC(K) (K-273.15)
          const float NTC_Beta = 4311;
          const float Vref     = 2.048;
          const float ADCmax   = (32767*psu.vcc_3V3/2.048);
          const float Rref     = 100000; // Ohms
          const float Tref     = 25;     // C
          float R = (Rref*Vref - Rref*(adc*Vref/ADCmax)) / (adc*Vref/ADCmax);
          float T = (CtoK(Tref)*NTC_Beta / log(Rref/R)) / (NTC_Beta / log(Rref/R) - CtoK(Tref)); // K
          T = KtoC(T);
          int i = constrain( T, 0, 99 );
          if (i != ui.temp)
          { ui.temp = i;
            if (ui.mode == UI_MODE_MAIN) // only on main screen
              paintTempIndicator( true );
          }
        }
        break;

        case 3: // 3V3 bus
        { // full scale is 32767 @ 2.048V
          float v = 2.048*adc/32767;
          // 20K/20K voltage divider
          psu.vcc_3V3 = v*(20000+20000)/20000;        
        }        
      }
          
      // bump state machine
      adc_scan = (adc_scan+1) % 4;
  
      // start next conversion
      uint16_t cmd = ADS1015_REG_CONFIG_CQUE_NONE    | // Disable the comparator (default val)
                     ADS1015_REG_CONFIG_CLAT_NONLAT  | // Non-latching (default val)
                     ADS1015_REG_CONFIG_CPOL_ACTVLOW | // Alert/Rdy active low   (default val)
                     ADS1015_REG_CONFIG_CMODE_TRAD   | // Traditional comparator (default val)
                     ADS1015_REG_CONFIG_DR_1600SPS   | // 1600 samples per second (default)
                     ADS1015_REG_CONFIG_MODE_SINGLE  | // Single-shot mode (default)
                     GAIN_TWO                        | // 2x gain   +/- 2.048V  1 bit = 1mV
                     ((adc_scan+4) << 12)            | // mux
                     ADS1015_REG_CONFIG_OS_SINGLE;     // start conversion
      writeRegister16(ADS1115_ADDR, ADS1015_REG_POINTER_CONFIG, cmd);
    }
  }

//-- update display

  if (ui.cv_display_pend) paintSetV( true );
  if (ui.cc_display_pend) paintSetI( true );

  { int n = calcCCCV();
    if (n != ui.cccv)
    { ui.cccv = n;
      paintCCCV( true );
    }
  }
  
//-- save settings to EEPROM  

  if (ui.update_pending)
    if (millis() - ui.update_tm >= AUTOSAVE_MS)
    {
      // copy ui_config/ui_settings -> ee_config/ee_settings -> EEPROM
      saveChanges(); 
      ui.update_pending = false;

      // restore set point colors
      if (ui.v_altcolor)
      { ui.v_altcolor = false;
        ui.cv_display_pend = true;
      }
      if (ui.c_altcolor)
      { ui.c_altcolor = false;
        ui.cc_display_pend = true;
      }
    }
}
