//------------------------------------------------------
// Stereo OScope
// 2023.07.04 RSP
// 2023.07.11 added XY plot mode
//
// Target: RP2040
//         240x240 TFT display (GC8A01)
//         Switches for Sweep Speed, Gain, Input Select, and Mode
//
// Description:
//   Displays stereo audio as oscilloscope wave, FFT graph, or XY plot.
//------------------------------------------------------

//#define DEBUG_CONSOLE // enable serial monitor diagnostics

// analog pins
#define ADC0_PIN   26 // L channel
#define ADC1_PIN   27 // R channel

// digital pins
#define SWEEP1_PIN  7 // sweep speed switch
#define SWEEP2_PIN  8
#define SWEEP3_PIN  9
#define SWEEP4_PIN 10
#define MODE_PIN   11 // mode button
#define GAINA_PIN  12 // gain switch
#define GAINB_PIN  13
#define SEL_L_PIN  14 // input select switch
#define SEL_R_PIN  15
#define GAIN1_PIN  22 // gain PGA control
#define GAIN2_PIN  28

// controls 

#define MODE_OSCOPE 0
#define MODE_FFT    1
#define MODE_XY     2
uint scope_mode     = MODE_OSCOPE;
uint new_scope_mode = MODE_OSCOPE;

uint sweep_mode     = 1; // 0=slowest
uint new_sweep_mode = 1;

uint gain_mode      = 1; // medium gain (consumer line level)
uint new_gain_mode  = 1;

uint input_mode     = 2; // 0=left, 1=right, 2=both
uint new_input_mode = 2;

boolean freeze_mode;     // display stutter mode

//------------------------------------------------------
// 240x240 Display (GC9A01)
/* 
  Library customizations...
    User_Setup_Select.h
    Setup200_GC9A01.h
      #define TFT_BL    16           // LED back-light control pin
      #define TFT_BACKLIGHT_ON HIGH  // Level to turn ON back-light (HIGH or LOW)
      #define TFT_MOSI  19
      #define TFT_SCLK  18
      #define TFT_CS    17  // Chip select control pin
      #define TFT_DC    20  // Data Command control pin
      #define TFT_RST   21  // Reset pin 
*/
#include "Free_Fonts.h"
#include "SPI.h"
#include "TFT_eSPI.h"
#include "font_Sat_32.h" // http://oleddisplay.squix.ch/#/home
#define CF_S32 &Satisfy_Regular_32 // title font
#define SCREEN_WIDTH  240
#define SCREEN_HEIGHT 240
TFT_eSPI tft = TFT_eSPI(SCREEN_WIDTH,SCREEN_HEIGHT); // Using hardware SPI
TFT_eSprite spr(&tft);   // image buffer
uint16_t   *pspr;         //   dynamically allocated
 
#define TRACE_L  TFT_RED    // left channel color
#define TRACE_R  TFT_WHITE  // right channel color
#define TRACE_XY TFT_ORANGE // XY plot color

//------------------------------------------------------
// ADC input

#include "hardware/adc.h"
#include "hardware/dma.h"
uint               adc_dma_ch;
dma_channel_config cfg;

#include "arduinoFFT.h"
const uint16_t samples = 1024; // This value MUST ALWAYS be a power of 2
double vReal[samples];
double vImag[samples];
arduinoFFT FFT = arduinoFFT(); 
#define FFT_SCALE 8            // magnitude divisor

// audio buffer needs 2*SCREEN_WIDTH for Oscope, 1024 for FFT
#define OSCOPE_DEPTH  (SCREEN_WIDTH*2*2) // X2 for overscan, X2 for stereo
#define FFT_DEPTH     (1024*2)           // always sample both channels
uint8_t capture_buf[1024*2]; // buffer large enough for OSCOPE_DEPTH or FFT_DEPTH

// ADC period of samples will be (1 + div) cycles on average. 
//   Note it takes 96 cycles to perform a conversion, so any period less than that will be clamped to 96.
//   ADC cycle clk = 48 MHz
float adc_clkdiv[5] = // n = (48MHz / 2*samp_rate) - 1, sample_rate = (48MHz / (n+1)) / 2
{       //                                             FFT resolution   Range
  4999, // 50 msec sweep, 4.8KHz X2 = 9.6KHz ADC rate    9.375 Hz       2250 Hz
  1999, // 20 msec                                      23.4375 Hz      5625 Hz
   999, // 10 msec                                      46.875 Hz      11250 Hz
   499, //  5 msec sweep,  48KHz X2 = 96KHz ADC rate    93.75 Hz       22500 Hz
   199  //  2 msec sweep                               234.375 Hz      56250 Hz
//  99  //  1 msec sweep, 240KHz X2 = 480KHz ADC rate  468.75 Hz
};

// OScope trigger on rising edge
const uint trigger_level = 127+13; // virtual ground is ~127

void setGain() // set input gain by enabling feedback resistors
{ // from gain_mode
  // gain 2 is highest, no extra feedback resistors enabled
  digitalWrite( GAIN1_PIN, gain_mode == 1 ); // 1=medium gain
  digitalWrite( GAIN2_PIN, gain_mode == 0 ); // 0=lowest gain
}

void configureDMA() // prepare DMA for ADC transfer
{ // requires adc_dma_ch
  cfg = dma_channel_get_default_config(adc_dma_ch);
  // Reading from constant address, writing to incrementing byte addresses
  channel_config_set_transfer_data_size(&cfg, DMA_SIZE_8);
  channel_config_set_read_increment(&cfg, false);
  channel_config_set_write_increment(&cfg, true);
  // Pace transfers based on availability of ADC samples
  channel_config_set_dreq(&cfg, DREQ_ADC);
  // configure and trigger channel (enable DMA on ADC reads)
  dma_channel_configure(adc_dma_ch, &cfg,
      capture_buf,    // dst
      &adc_hw->fifo,  // src
      scope_mode ? FFT_DEPTH : OSCOPE_DEPTH, // transfer count
      true);          // start immediately
}

//------------------------------------------------------
// Processor Core 0
//------------------------------------------------------

// run single core for initialization
volatile boolean hold_core_1 = true;

void setup() 
{
#ifdef DEBUG_CONSOLE  
  Serial.begin(115200); 
  while (!Serial && millis() < 10000UL);
  Serial.println("core 0 start"); 
#endif

  pinMode( SWEEP1_PIN, INPUT_PULLUP );
  pinMode( SWEEP2_PIN, INPUT_PULLUP );
  pinMode( SWEEP3_PIN, INPUT_PULLUP );
  pinMode( SWEEP4_PIN, INPUT_PULLUP );
  pinMode( MODE_PIN,   INPUT_PULLUP );
  pinMode( GAINA_PIN,  INPUT_PULLUP );
  pinMode( GAINB_PIN,  INPUT_PULLUP );
  pinMode( SEL_L_PIN,  INPUT_PULLUP );
  pinMode( SEL_R_PIN,  INPUT_PULLUP );

  pinMode( GAIN1_PIN, OUTPUT );
  pinMode( GAIN2_PIN, OUTPUT );
  setGain(); // set gain pins for gain_mode

  // set up LCD display
  tft.init(); // same as begin()
  tft.fillScreen(TFT_BLACK); // clear screen immediately (non-DMA)
  // 240x240 = 115200 byte 'sprite' display buffer (this sprite is the whole screen)
  pspr = (uint16_t*)spr.createSprite(SCREEN_WIDTH, SCREEN_HEIGHT);
  tft.initDMA();
  tft.startWrite(); // TFT chip select held low permanently for DMA
  
  // set up ADC
  adc_gpio_init(ADC0_PIN);
  adc_gpio_init(ADC1_PIN);
  adc_init();
  adc_set_round_robin(3); // 2 channel measurement : ch 0 & 1
  adc_select_input(0);    //   starting channel
  adc_fifo_setup(true,    // Enables write each conversion result to the FIFO
                 true,    // Enable DMA requests when FIFO contains data
                 1,       // Threshold for DMA requests/FIFO IRQ if enabled
                 false,   // If enabled, bit 15 of the FIFO contains error flag for each sample
                 true);   // Shift FIFO contents to be one byte in size (for byte DMA) - enables DMA to byte buffers
  adc_set_clkdiv(adc_clkdiv[sweep_mode]);
  
  // set up DMA for ADC data transfer
  adc_dma_ch = dma_claim_unused_channel(true);
  configureDMA();

  // press mode at power up to activate freeze mode
  freeze_mode = !digitalRead(MODE_PIN);

  // now safe to start 2'nd core
  hold_core_1 = false; // let setup1 run
  
  // splash screen
  spr.fillSprite(TFT_BLACK);
  spr.setTextColor( TFT_WHITE, TFT_BLACK );  
  spr.setTextDatum(TC_DATUM); 
  spr.setFreeFont(CF_S32);
  spr.drawString("OScope", SCREEN_WIDTH/2,80);
  spr.setFreeFont(FSS9);             
  spr.drawString("www.FearlessNight.com", SCREEN_WIDTH/2,130);
  tft.pushImageDMA(0,0, SCREEN_WIDTH,SCREEN_HEIGHT, pspr);
  delay(2000); 
  
  // gather first audio sample
  adc_run(true); // start free running ADC capture
}

// - - - - - - - - - - - - -
// core 0 loop

void loop(void)
{
  
//-- wait for audio DMA to finish

  dma_channel_wait_for_finish_blocking(adc_dma_ch);
  // DMA is stopped; stop ADC free run capture too
  adc_run(false);
  adc_fifo_drain(); // clean up the FIFO in case the ADC was still mid-conversion

//-- render audio sample buffer to sprite

  spr.fillSprite(TFT_BLACK); // clear the display buffer
  switch (scope_mode)
  {
    case MODE_OSCOPE:
      { // search for trigger (triggers on left channel only)
        uint trig; // L channel start in capture_buf
        for (trig=0; trig < OSCOPE_DEPTH/2-2; trig+=2)
          // look for positive crossing of the trigger level
          if (capture_buf[trig] <= trigger_level && capture_buf[trig+1] >= trigger_level
           && capture_buf[trig] <= capture_buf[trig+1]) break;
        if (trig >= OSCOPE_DEPTH/2-2) trig = 0; // no trigger found, auto trigger
        // plot
        //   L/R data is interleaved
        for (uint x=2; x < SCREEN_WIDTH; x+=2)
        {
          if (input_mode != 1) // left | both
          { // plot left channel
            uint y0 = map( capture_buf[trig+x-2], 0,255, 0,(SCREEN_HEIGHT-1) );
            uint y1 = map( capture_buf[trig+x],   0,255, 0,(SCREEN_HEIGHT-1) );
            if (y0 >= SCREEN_HEIGHT) y0 = SCREEN_HEIGHT;
            if (y1 >= SCREEN_HEIGHT) y1 = SCREEN_HEIGHT;
            spr.drawLine( x-1,y0, x,y1, TRACE_L );
          }
          if (input_mode != 0) // right | both
          { // plot right channel
            uint y0 = map( capture_buf[trig+x-1], 0,255, 0,(SCREEN_HEIGHT-1) );
            uint y1 = map( capture_buf[trig+x+1], 0,255, 0,(SCREEN_HEIGHT-1) );
            if (y0 >= SCREEN_HEIGHT) y0 = SCREEN_HEIGHT;
            if (y1 >= SCREEN_HEIGHT) y1 = SCREEN_HEIGHT;
            spr.drawLine( x-1,y0, x,y1, TRACE_R );
          }
        }
        break;
      }
      
    case MODE_FFT:
      if (input_mode != 1) // left | both
      { // plot left channel
        for (uint i=0; i < samples; i++)
        { vReal[i] = capture_buf[i*2]-128.0; // left channel data only
          vImag[i] = 0.0;
        }
        FFT.Windowing(vReal, samples, FFT_WIN_TYP_HAMMING, FFT_FORWARD); 
        FFT.Compute(vReal, vImag, samples, FFT_FORWARD);
        FFT.ComplexToMagnitude(vReal, vImag, samples); 
        for (uint x=0; x < SCREEN_WIDTH; x++)
        { int y = vReal[x];
              y /= FFT_SCALE;
          if (y >  SCREEN_HEIGHT/2-1) y =  SCREEN_HEIGHT/2-1;
          spr.drawLine( x,SCREEN_HEIGHT/2, x,SCREEN_HEIGHT/2-y, TRACE_L );
        }
      }
      if (input_mode != 0) // right | both
      { // plot right channel
        for (uint i=0; i < samples; i++)
        { vReal[i] = capture_buf[i*2+1]-128.0; // right channel data only
          vImag[i] = 0.0;
        }
        FFT.Windowing(vReal, samples, FFT_WIN_TYP_HAMMING, FFT_FORWARD); 
        FFT.Compute(vReal, vImag, samples, FFT_FORWARD);
        FFT.ComplexToMagnitude(vReal, vImag, samples); 
        for (uint x=0; x < SCREEN_WIDTH; x++)
        { int y = vReal[x];
              y /= FFT_SCALE;
          if (y >  SCREEN_HEIGHT/2-1) y =  SCREEN_HEIGHT/2-1;
          spr.drawLine( x,SCREEN_HEIGHT/2, x,SCREEN_HEIGHT/2+y, TRACE_R );
        }
      }
      break;

    case MODE_XY:
      {
        // search for trigger (triggers on left channel only)
        uint trig; // L channel start in capture_buf
        for (trig=0; trig < OSCOPE_DEPTH/2-2; trig+=2)
          // look for positive crossing of the trigger level
          if (capture_buf[trig] <= trigger_level && capture_buf[trig+1] >= trigger_level
           && capture_buf[trig] <= capture_buf[trig+1]) break;
        if (trig >= OSCOPE_DEPTH/2-2) trig = 0; // no trigger found, auto trigger
        // plot
        //   L/R data is interleaved
        uint x_prev =     capture_buf[trig];
        uint y_prev = 255-capture_buf[trig+1];
        for (uint i=2; i < SCREEN_WIDTH; i+=2)
        {
          uint x = map(     capture_buf[trig+i],   0,255, 0,(SCREEN_HEIGHT-1) );
          uint y = map( 255-capture_buf[trig+i+1], 0,255, 0,(SCREEN_HEIGHT-1) );
          if (x >= SCREEN_HEIGHT) x = SCREEN_HEIGHT;
          if (y >= SCREEN_HEIGHT) y = SCREEN_HEIGHT;
          spr.drawLine( x_prev,y_prev, x,y, TRACE_XY );
          x_prev = x;
          y_prev = y;
        }
      }
  }
  
//-- start next audio sample collection
  
  // but first, update settings from controls
  if (input_mode != new_input_mode)
    input_mode = new_input_mode;

  if (sweep_mode != new_sweep_mode)
  { sweep_mode = new_sweep_mode;
    adc_set_clkdiv(adc_clkdiv[sweep_mode]); // set ADC read rate
  }
  
  if (gain_mode != new_gain_mode)
  { gain_mode = new_gain_mode;
    setGain(); // set PGA
  }

  if (scope_mode != new_scope_mode)
  { scope_mode = new_scope_mode;
    spr.fillSprite(TFT_BLACK);
    spr.setTextDatum(TC_DATUM);  
    spr.setFreeFont(FSSB18);             
    spr.drawString( scope_mode == 0 ? "Oscilloscope" : (scope_mode == 1 ? "FFT" : "XY"), SCREEN_WIDTH/2,100);
    tft.pushImageDMA(0,0, SCREEN_WIDTH,SCREEN_HEIGHT, pspr);
    delay(1000);    
    spr.fillScreen(TFT_BLACK); // abort last frame
  }

  // okay, now start ADC
  configureDMA();       // get DMA ready for ADC
  adc_select_input(0);  // synchronize ADC
  adc_run(true);        // start free running ADC capture

//-- DMA sprite -> display, while collecting audio samples

  do{ // skip if freezing display
      if (freeze_mode)
      { static uint32_t tm = millis();
        if (millis() - tm < 5000) break; else tm = millis();
      }
      tft.pushImageDMA(0,0, SCREEN_WIDTH,SCREEN_HEIGHT, pspr);
      while(tft.dmaBusy()) ; // wait for DMA to complete
    } while (0); // do once
 
//-- diagnostic FPS readout

#ifdef DEBUG_CONSOLE  
  { static uint32_t tm = millis();
    static uint     frames = 0;
    frames++;
    if (millis() - tm >= 5000)
    { Serial.print("FPS: "); Serial.println(frames/5.0);
      tm = millis();
      frames = 0;
    }
  }
#endif  
}

//------------------------------------------------------
// Processor Core 1 = user control inputs
//------------------------------------------------------

void setup1()
{
  while (hold_core_1) delay(1); // wait for core0 to finish
}

// - - - - - - - - - - - - -

void loop1(void)
{
#ifdef DEBUG_CONSOLE  
  { int c = Serial.read();
    switch (c)
    { case -1: break;
      // photography
      case 'z': freeze_mode = !freeze_mode; break;
      // mode
      case 'f': new_scope_mode = 1; break;
      case 'o': new_scope_mode = 0; break;
      case 'x': new_scope_mode = 2; break;
      // sweep speed
      case '1': new_sweep_mode = 0; break;
      case '2': new_sweep_mode = 1; break;
      case '3': new_sweep_mode = 2; break;
      case '4': new_sweep_mode = 3; break;
      case '5': new_sweep_mode = 4; break;
      // gain
      case 'g': new_gain_mode = 0; break;
      case 'h': new_gain_mode = 1; break;
      case 'i': new_gain_mode = 2; break;
      // input selection
      case 'l': new_input_mode = 0; break;
      case 'r': new_input_mode = 1; break;
      case 'b': new_input_mode = 2; break;
    }
  }
#else // monitor switches
  
  { // sweep speed select
    uint nmode = 0;
    if (digitalRead(SWEEP1_PIN) == LOW) nmode = 1;
    if (digitalRead(SWEEP2_PIN) == LOW) nmode = 2;
    if (digitalRead(SWEEP3_PIN) == LOW) nmode = 3;
    if (digitalRead(SWEEP4_PIN) == LOW) nmode = 4;
    new_sweep_mode = nmode;
  }

  { // gain low/med/high select
    uint nmode = 0;
    if (digitalRead(GAINA_PIN) == LOW) nmode = 1;
    if (digitalRead(GAINB_PIN) == LOW) nmode = 2;
    new_gain_mode = nmode;
  }    
  
  { // input L/R/Both select 
    uint nmode = 2; // default is both
    if (digitalRead(SEL_L_PIN) == LOW) nmode = 0;
    if (digitalRead(SEL_R_PIN) == LOW) nmode = 1;
    new_input_mode = nmode;
  }
  
  { // mode button
    static boolean  btn_state = false;
    static uint32_t btn_tm; // for debouncing
    if (digitalRead(MODE_PIN) == LOW)
    { // button is pressed
      if (!btn_state)
      { // button press event
        btn_state = true;
        new_scope_mode = (new_scope_mode+1) % 3; // bump mode
      }
      btn_tm = millis(); // [re]start debounce
    }
    else // button is not pressed
      if (btn_state)
        if (millis() - btn_tm > 50) // 50 msec debounce
          btn_state = false; // debounce ended, button release event
  }  
#endif
}
