//-------------------------------------------------------------------
// Stoplight Simulator
// 2024.04.28 RSP
// Target: ATmega328PB (RSP's breakout board)
//     (4) WS2812 strings for stoplights
//     (4) HT16K33 dual 7seg walk timers (8 x 2 digits)
//     Using MiniCore for Wire1 support
// Operation:
//    Speed switch freezes or accelerates (4X) the stoplight timing.
// Serial Monitor Debug:
//    #+ | #- = set/coear bit #
//    v       = toggle verbose output
//    f       = set fallback mode
//    c       = clear errors & fallback mode
//    0..3    = set stoplight (for identificaton purposes)
// Processing UI (on Serial1):
//    processing => controller (no CR and '#' is binary)
//      S# sensor bits 
//      X# speed 0,1,2 = 1X, 2X, 4X 
//    controller => processing (all ASCII, CR terminated lines)
//      A#,#,#,# thru light states; 0=red, 1=grn, 2=yel, 3=off
//      B#,#,#,# left turn light states; 0=red, 1=grn, 2=yel, 3=off
//      D#,#,##  walk countdowns; pole,unit,count
//      C#       pedestrian request cycle clear, # = direction 0|1
//-------------------------------------------------------------------

//#define processing // Processing UI on Serial1 (disables hardware traffic & pedestrian sensors)

#define HAZIO // HT16K33's on I2C (walk time displays)

// debugging on Serial Monitor
boolean verbose = true; // togglable in Serial Monitor
//#define VBUG(code) {if(verbose){code}} // verbose debug on Serial Monitor
#define VBUG(code) {;} // minimum memory
const String COLOR_NAME[4] = { "RED", "GRN", "YEL", "OFF" };
const String SENSOR_DESC[8] = { "EW Ped", "NS Ped", "W Left Turn", "W Traffic", "N Traffic", "S Traffic", "E Traffic", "E Left Turn" };
int tick = 0;

// time warp based on speed switch
uint32_t frame_msec = 1000; // 1000 for real time

// coroutine macros
#define coBegin { static int _state_ = 0; switch(_state_) { case 0:;
#define coEnd     _state_ = 0; }}
#define coWaitWhile(expr) { _state_ = __LINE__; return;  case __LINE__: if (expr) return; }
#define coDelay(msec) { static uint32_t tm; _state_ = __LINE__; tm=millis(); return; case __LINE__: if (millis()-tm < msec) return; }
#define coDelayWhile(expr,msec) { static uint32_t tm; _state_ = __LINE__; tm=millis(); return; case __LINE__: if ((expr) && millis()-tm < msec) return; }
#define coDelayWarped(msec) { static uint32_t tm; _state_ = __LINE__; tm=millis(); return; case __LINE__: if (millis()-tm < msec*frame_msec/1000UL || frame_msec == 0) return; }
#define coYield       { _state_ = __LINE__; return;  case __LINE__: ; }

//-------------------------------------------------------------------
// 4 stoplight neopixel strings

#define STOPLIGHT_BRIGHTNESS 20
#include <Adafruit_NeoPixel.h>
Adafruit_NeoPixel stoplight[4] = 
{ {4, 10, NEO_GRB + NEO_KHZ800},
  {4, 11, NEO_GRB + NEO_KHZ800},
  {4, 12, NEO_GRB + NEO_KHZ800},
  {4, 13, NEO_GRB + NEO_KHZ800} };
const byte LAMP_MAP[4] = { 1,3,2,0 }; // actual hardware arrangement

//-------------------------------------------------------------------
// 4 walk time displays

#include "HT16K33.h"
stoplight_7seg_HT16K33 walk_display[4] = {HT16K33_DEFAULT_I2C_ADDR, HT16K33_DEFAULT_I2C_ADDR+1, HT16K33_DEFAULT_I2C_ADDR+2, HT16K33_DEFAULT_I2C_ADDR+3};

//-------------------------------------------------------------------
// traffic sensors & pedestrian buttons

#define MASK(b) (1 << (b))
uint8_t sensors;
#define EW_PED_BIT  0
#define NS_PED_BIT  1
#define N_THRU_BIT  4
#define S_THRU_BIT  5
#define E_THRU_BIT  6
#define W_THRU_BIT  3
#define E_TURN_BIT  7
#define W_TURN_BIT  2

// speed switch
#define SWA 15 // A1
#define SWB 14 // A0

//-------------------------------------------------------------------
// error handler

#define PWR_LED_PIN 16 // blinks error code
#define BLINK_DELAY_MS 1000
#define BLINK_ON_MS     100
#define BLINK_OFF_MS    500
uint8_t fault_code = 0;
#define E_PCF8575 1
#define E_HT16K33 2
#define E_IO      3
#define E_SAFETY  4
void fault( uint8_t blinks )
{
  fault_code = blinks;
VBUG(Serial.print(F("FAULT: ")); Serial.println(blinks);)
}
void faultTask() // blink error code in real time
{ static uint8_t count;
  coBegin
    if (fault_code)
    { count = fault_code;
      digitalWrite( PWR_LED_PIN, LOW );
      coDelayWhile( fault_code != 0, BLINK_DELAY_MS );
      for (; count; count--)
      { digitalWrite( PWR_LED_PIN, HIGH );
        coDelayWhile( fault_code != 0, BLINK_ON_MS );
        digitalWrite( PWR_LED_PIN, LOW );
        coDelayWhile( fault_code != 0, BLINK_OFF_MS );
      }
    }
  coEnd
}

//-------------------------------------------------------------------
// stoplight application

boolean fallback_mode = false; // safe mode
byte direction = 0;  // traffic direction; 0=N+S, 1=E+W
uint16_t cycle_sec;
enum light_state { _RED, _GRN, _YEL, _OFF };
#define N_LIGHT 0
#define E_LIGHT 1
#define S_LIGHT 2
#define W_LIGHT 3
struct stoplight_control
{ 
  light_state color;
  uint16_t start_sec; // offset from cycle_sec
  uint16_t end_sec;
  boolean  event;     // end indicator
  int8_t   traffic_sensor_bit;
};                            //     N                             E                            S                             W
stoplight_control main_grn[4] = { {_RED,0,0,false, N_THRU_BIT}, {_RED,0,0,false,E_THRU_BIT}, {_RED,0,0,false, S_THRU_BIT}, {_RED,0,0,false,W_THRU_BIT} };
stoplight_control leftturn[4] = { {_RED,0,0,false,-1},          {_RED,0,0,false,E_TURN_BIT}, {_RED,0,0,false,-1},          {_RED,0,0,false,W_TURN_BIT} };
struct
{ uint8_t count;
  uint8_t pole1; // display location
  uint8_t unit1;
  uint8_t pole2;
  uint8_t unit2;
} walk_signal[4] = { {0,2,1,3,0}, {0,1,1,2,0}, {0,0,1,1,0} ,{0,0,0,3,1} };

boolean ped_request_pending[2]; // pedestrian crossing button pressed

#define LEFT_MAX_MS           60000UL // max left turn signal on time
#define YEL_MSEC               5000UL // yellow is always 5 seconds
#define FALLBACK_BLINK_ON_TIME 1000UL // msec
#define FALLBACK_BLINK_OFF_TIME 500UL // msec
#define MIN_GREEN_TIME           10U  // seconds
#define MAX_GREEN_TIME          120U  // seconds
#define GREEN_EXTEND_TIME        10U  // traffic present
#define PEDESTRIAN_CROSSING_TIME 30U  // seconds
#define CROSSTRAFFIC_GREEN_MAX_TIME 60U
#define WALK_BRIGHTNESS 1 // walk time LED driver brightness, 0..15

//-------------------------------------------------------------------

void setup()
{
Serial.begin(115200); // debug
while (!Serial && (millis() <= 10000)) delay(10);
VBUG(Serial.println(F("Started!"));)

#ifdef processing
  Serial1.begin(115200);
#endif
  
  // pull ups
  PORTD = 0xFC;
  PORTB = 0x03;

  pinMode( SWA, INPUT_PULLUP );
  pinMode( SWB, INPUT_PULLUP );
  
  // power LED on
  pinMode( PWR_LED_PIN, OUTPUT );  digitalWrite( PWR_LED_PIN, HIGH );

#ifdef HAZIO  
  // start I2C & verify devices are on the bus
  MYWIRE.begin();
  MYWIRE.beginTransmission(HT16K33_DEFAULT_I2C_ADDR+0); if (MYWIRE.endTransmission()) fault(E_HT16K33);
  MYWIRE.beginTransmission(HT16K33_DEFAULT_I2C_ADDR+1); if (MYWIRE.endTransmission()) fault(E_HT16K33);
  MYWIRE.beginTransmission(HT16K33_DEFAULT_I2C_ADDR+2); if (MYWIRE.endTransmission()) fault(E_HT16K33);
  MYWIRE.beginTransmission(HT16K33_DEFAULT_I2C_ADDR+3); if (MYWIRE.endTransmission()) fault(E_HT16K33);
  // initialize displays
  for (uint8_t i=0; i < 4; i++)
  { if (walk_display[i].begin( WALK_BRIGHTNESS )) fault(E_HT16K33);
    walk_signal[i].count = 0;
    doPedestrianLight(i);
  }
#else
sensors = 0xFF;
#endif

  // initialize WS2812 strings
  for (uint8_t i=0; i < 4; i++)
  { stoplight[i].begin();
    stoplight[i].show();
  }
}

//-------------------------------------------------------------------

void loop() 
{
task1(); // debug

#ifndef processing
  // speed switch
  if (!digitalRead(SWB))      frame_msec =    0; //   down position = freeze
  else if (!digitalRead(SWA)) frame_msec =  250; //     up position = X4 speed
  else                        frame_msec = 1000; // middle position = X1 speed
#endif
  // 1 second tick, precision timekeeping
  static uint32_t tm = millis();
  if (frame_msec)
    if (millis() - tm >= frame_msec)
    { tm += frame_msec;
      cycle_sec++;
    }

  // read sensors
#ifndef processing
  sensors = (PIND & 0xFC) | (PINB & 0x03);
#endif

  if (fallback_mode)
  { // simply blink reds
    fallbackTask();
    outputLightsTask();         // stoplight output
  }
  else // operating normally
  { // run stoplight control tasks
    stopLightsTask();             // stoplight sequencing
    monitorSensorsTask();         // monitor traffic & pedestrian sensors
    Nstoplight();                 // thru light sequencing
    Estoplight();
    Sstoplight();
    Wstoplight();
    NleftTurnSignal();            // left turn signal sequencing
    EleftTurnSignal();
    SleftTurnSignal();
    WleftTurnSignal();
    outputLightsTask();           // stoplight output
    NpedestrianLight();           // pedestrian countdown output
    EpedestrianLight();           
    SpedestrianLight();           
    WpedestrianLight();           
    overwatchTask();              // safety checks
  }
    
  faultTask(); // housekeeping (flash the fault LED)
  
  // simulate pedestrian and traffic sensor input from serial monitor
  { static uint8_t bit_to_set = 0;
    int c = Serial.read();
    switch (c)
    { 
      case '0': stoplight[0].setPixelColor( 0, stoplight[0].Color(STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS));stoplight[0].show(); break;
      case '1': stoplight[1].setPixelColor( 1, stoplight[0].Color(STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS));stoplight[1].show(); break;
      case '2': stoplight[2].setPixelColor( 2, stoplight[0].Color(STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS));stoplight[2].show(); break;
      case '3': stoplight[3].setPixelColor( 3, stoplight[0].Color(STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS));stoplight[3].show(); break;
      
      case '+': sensors &= ~MASK(bit_to_set); Serial.print("Bit ON "); Serial.println(bit_to_set); debugSensors(); break;
      case '-': sensors |=  MASK(bit_to_set); Serial.print("Bit OFF "); Serial.println(bit_to_set); debugSensors(); break;
      case 'v': case 'V': verbose = !verbose; Serial.println(verbose ? "Verbose activated":"Shutting up, sir"); break;
      case 'c': case 'C': fault_code = 0; fallback_mode = false; for (uint8_t i=0; i < 4; i++) main_grn[i].color=_RED; digitalWrite( PWR_LED_PIN, HIGH ); break;
      case 'f': case 'F': 
                fallback_mode = true;
                for (uint8_t i=0; i < 4; i++) main_grn[i].color = leftturn[i].color = _RED;
                for (uint8_t i=0; i < 4; i++) { walk_signal[i].count = 0;  doPedestrianLight(i); }
                break;
      case '?': Serial.println(F("Stoplight Simulator"));
                Serial.println(F("c  clear error"));
                Serial.println(F("v  toggle verbose"));
                Serial.println(F("#+ set sensor bit"));
                Serial.println(F("#- clr sensor bit"));
                debugSensors();
                break;
      default : if (c >= '0' && c <= '9') bit_to_set = c-'0';
    }

    // handle input from Processing
#ifdef processing    
    { static uint8_t processing_id = 0; // header char for data byta
      int c = Serial1.read();
      if (c >= 0)
        switch (processing_id)
        { case 'S': sensors = c ^ 0xFF; 
                    processing_id = 0;  break;
          case 'X': switch (c)
                    { case 2 : frame_msec =  250UL; break;  // 2 X4
                      case 1 : frame_msec =  500UL; break;  // 1 X2
                      default: frame_msec = 1000UL;         // 0 X1
                    }
                    processing_id = 0;  break;
          default : processing_id = c;
        }
    }
#endif
  }
}
void debugSensors()
{
  for (uint8_t i=0; i <= 7; i++)
  {
    Serial.print("["); Serial.print(i); Serial.print("] "); Serial.print((sensors & MASK(i)) ? "OFF ":"ON "); Serial.println(SENSOR_DESC[i]);
  }
}

//-------------------------------------------------------------------

void task1() // debug; prints elapsed seconds in current state + current cycle time
{ coBegin
VBUG(Serial.print("+"); Serial.print(tick++); Serial.print(" "); Serial.println(cycle_sec);)
    coDelayWarped(1000);
  coEnd
}

//-------------------------------------------------------------------

void overwatchTask() // detect conflict situations, set fallback mode
{
  static  uint8_t error_count = 0;
  boolean error_detected = false;
    
  // conflicting greens check
  if ((main_grn[N_LIGHT].color == _GRN || main_grn[S_LIGHT].color == _GRN)
   && (main_grn[E_LIGHT].color == _GRN || main_grn[W_LIGHT].color == _GRN))
    error_detected = true;

  // conflicting left turn check 
  if ((main_grn[N_LIGHT].color == _GRN && leftturn[S_LIGHT].color == _GRN)
   || (main_grn[E_LIGHT].color == _GRN && leftturn[W_LIGHT].color == _GRN)
   || (main_grn[S_LIGHT].color == _GRN && leftturn[N_LIGHT].color == _GRN)
   || (main_grn[W_LIGHT].color == _GRN && leftturn[E_LIGHT].color == _GRN))
    error_detected = true;
    
  // conflicting pedestrian check
  if ((main_grn[N_LIGHT].color == _GRN || main_grn[S_LIGHT].color == _GRN)
   && (walk_signal[N_LIGHT].count > 0 || walk_signal[S_LIGHT].count > 0))
    error_detected = true;
  if ((main_grn[E_LIGHT].color == _GRN || main_grn[W_LIGHT].color == _GRN)
   && (walk_signal[E_LIGHT].count > 0 || walk_signal[W_LIGHT].count > 0))
    error_detected = true;

  // any errors found ?
  if (!error_detected) error_count = 0;
  else if (++error_count > 5)
  { fault(E_SAFETY);
    fallback_mode = true;
    for (uint8_t i=0; i < 4; i++) main_grn[i].color = leftturn[i].color = _RED;
    for (uint8_t i=0; i < 4; i++) { walk_signal[i].count = 0;  doPedestrianLight(i); }
VBUG(Serial.print(F("OVERWATCH DETECTED FAULT"));)
  }
}

//-------------------------------------------------------------------

void NleftTurnSignal() // run the left turn signal
{
  coBegin
    if (leftturn[N_LIGHT].traffic_sensor_bit != -1) // has left turn signal?
      if (leftturn[N_LIGHT].color == _GRN)
      { 
VBUG(Serial.print("Left "); Serial.print(N_LIGHT); Serial.print(" "); Serial.println("GRN");)        
        coWaitWhile( leftturn[N_LIGHT].end_sec > cycle_sec );
        leftturn[N_LIGHT].color = _YEL;
VBUG(Serial.print("Left "); Serial.print(N_LIGHT); Serial.print(" "); Serial.println("YEL");)
        coDelayWarped( YEL_MSEC );  // wait for yellow time
        leftturn[N_LIGHT].color = _RED;
        leftturn[N_LIGHT].event = true;
VBUG(Serial.print("Left "); Serial.print(N_LIGHT); Serial.print(" "); Serial.println("RED");)        
      }
  coEnd
}
void EleftTurnSignal() // run the left turn signal
{
  coBegin
    if (leftturn[E_LIGHT].traffic_sensor_bit != -1) // has left turn signal?
      if (leftturn[E_LIGHT].color == _GRN)
      { 
VBUG(Serial.print("Left "); Serial.print(E_LIGHT); Serial.print(" "); Serial.println("GRN");)
        coWaitWhile( leftturn[E_LIGHT].end_sec > cycle_sec );
        leftturn[E_LIGHT].color = _YEL;
VBUG(Serial.print("Left "); Serial.print(E_LIGHT); Serial.print(" "); Serial.println("YEL");)
        coDelayWarped( YEL_MSEC );  // wait for yellow time
        leftturn[E_LIGHT].color = _RED;
        leftturn[E_LIGHT].event = true;
VBUG(Serial.print("Left "); Serial.print(E_LIGHT); Serial.print(" "); Serial.println("RED");)
      }
  coEnd
}
void SleftTurnSignal() // run the left turn signal
{
  coBegin
    if (leftturn[S_LIGHT].traffic_sensor_bit != -1) // has left turn signal?
      if (leftturn[S_LIGHT].color == _GRN)
      { 
VBUG(Serial.print("Left "); Serial.print(S_LIGHT); Serial.print(" "); Serial.println("GRN");)
        coWaitWhile( leftturn[S_LIGHT].end_sec > cycle_sec );
        leftturn[S_LIGHT].color = _YEL;
VBUG(Serial.print("Left "); Serial.print(S_LIGHT); Serial.print(" "); Serial.println("YEL");)
        coDelayWarped( YEL_MSEC );  // wait for yellow time
        leftturn[S_LIGHT].color = _RED;
        leftturn[S_LIGHT].event = true;
VBUG(Serial.print("Left "); Serial.print(S_LIGHT); Serial.print(" "); Serial.println("RED");)
      }
  coEnd
}
void WleftTurnSignal() // run the left turn signal
{
  coBegin
    if (leftturn[W_LIGHT].traffic_sensor_bit != -1) // has left turn signal?
      if (leftturn[W_LIGHT].color == _GRN)
      { 
VBUG(Serial.print("Left "); Serial.print(W_LIGHT); Serial.print(" "); Serial.println("GRN");)
        coWaitWhile( leftturn[W_LIGHT].end_sec > cycle_sec );
        leftturn[W_LIGHT].color = _YEL;
VBUG(Serial.print("Left "); Serial.print(W_LIGHT); Serial.print(" "); Serial.println("YEL");)
        coDelayWarped( YEL_MSEC );  // wait for yellow time
        leftturn[W_LIGHT].color = _RED;
        leftturn[W_LIGHT].event = true;
VBUG(Serial.print("Left "); Serial.print(W_LIGHT); Serial.print(" "); Serial.println("RED");)
      }
  coEnd
}

//-------------------------------------------------------------------

void Nstoplight() // run the stoplight
{
  coBegin
    if (main_grn[N_LIGHT].color == _GRN)
    { 
VBUG(Serial.print("Thru "); Serial.print(N_LIGHT); Serial.print(" "); Serial.println("GRN");)
      coWaitWhile( main_grn[N_LIGHT].end_sec > cycle_sec) // || leftturn[S_LIGHT].color == _GRN );
      main_grn[N_LIGHT].color = _YEL;
VBUG(Serial.print("Thru "); Serial.print(N_LIGHT); Serial.print(" "); Serial.println("YEL");)
      coDelayWarped( YEL_MSEC );  // wait for yellow time
      main_grn[N_LIGHT].color = _RED;
VBUG(Serial.print("Thru "); Serial.print(N_LIGHT); Serial.print(" "); Serial.println("RED");)
    }
  coEnd
}

void Estoplight() // run the stoplight
{
  coBegin
    if (main_grn[E_LIGHT].color == _GRN)
    { 
VBUG(Serial.print("Thru "); Serial.print(E_LIGHT); Serial.print(" "); Serial.println("GRN");)
      coWaitWhile( main_grn[E_LIGHT].end_sec > cycle_sec) // || leftturn[W_LIGHT].color == _GRN );
      main_grn[E_LIGHT].color = _YEL;
VBUG(Serial.print("Thru "); Serial.print(E_LIGHT); Serial.print(" "); Serial.println("YEL");)
      coDelayWarped( YEL_MSEC );  // wait for yellow time
      main_grn[E_LIGHT].color = _RED;
VBUG(Serial.print("Thru "); Serial.print(E_LIGHT); Serial.print(" "); Serial.println("RED");)
   }
  coEnd
}

void Sstoplight() // run the stoplight
{
  coBegin
    if (main_grn[S_LIGHT].color == _GRN)
    { 
VBUG(Serial.print("Thru "); Serial.print(S_LIGHT); Serial.print(" "); Serial.println("GRN");)
      coWaitWhile( main_grn[S_LIGHT].end_sec > cycle_sec) // || leftturn[N_LIGHT].color == _GRN );
      main_grn[S_LIGHT].color = _YEL;
VBUG(Serial.print("Thru "); Serial.print(S_LIGHT); Serial.print(" "); Serial.println("YEL");)
      coDelayWarped( YEL_MSEC );  // wait for yellow time
      main_grn[S_LIGHT].color = _RED;
VBUG(Serial.print("Thru "); Serial.print(S_LIGHT); Serial.print(" "); Serial.println("RED");)
    }
  coEnd
}

void Wstoplight() // run the stoplight
{
  coBegin
    if (main_grn[W_LIGHT].color == _GRN)
    { 
VBUG(Serial.print("Thru "); Serial.print(W_LIGHT); Serial.print(" "); Serial.println("GRN");)
      coWaitWhile( main_grn[W_LIGHT].end_sec > cycle_sec) // || leftturn[E_LIGHT].color == _GRN );
      main_grn[W_LIGHT].color = _YEL;
VBUG(Serial.print("Thru "); Serial.print(W_LIGHT); Serial.print(" "); Serial.println("YEL");)
      coDelayWarped( YEL_MSEC );  // wait for yellow time
      main_grn[W_LIGHT].color = _RED;
VBUG(Serial.print("Thru "); Serial.print(W_LIGHT); Serial.print(" "); Serial.println("RED");)
    }
  coEnd
}

//-------------------------------------------------------------------

void fallbackTask() // run fallback stoplight sequence
{
  coBegin // all blinking red
    main_grn[N_LIGHT].color = leftturn[N_LIGHT].color = main_grn[S_LIGHT].color = leftturn[S_LIGHT].color =
    main_grn[E_LIGHT].color = leftturn[E_LIGHT].color = main_grn[W_LIGHT].color = leftturn[W_LIGHT].color = _RED;
    coDelayWarped(FALLBACK_BLINK_ON_TIME);
    main_grn[N_LIGHT].color = leftturn[N_LIGHT].color = main_grn[S_LIGHT].color = leftturn[S_LIGHT].color =
    main_grn[E_LIGHT].color = leftturn[E_LIGHT].color = main_grn[W_LIGHT].color = leftturn[W_LIGHT].color = _OFF;
    coDelayWarped(FALLBACK_BLINK_OFF_TIME);
  coEnd
}

void stopLightsTask() // run normal stoplight sequence (green cycle)
{ 
  static boolean ped_mode;
  
  coBegin
    cycle_sec = 0; // start timing reference

    // capture ped request
    ped_mode = ped_request_pending[direction];  ped_request_pending[direction] = false;
    // send to Processing
#ifdef processing
    Serial1.write('C'); Serial1.write(direction+'0'); Serial1.println();
#endif
    
tick = 0;    
    // figure out which greens are on initially
    uint16_t min_main_grn;
    if (!ped_mode) // left turn disabled for ped crossing
    {
      min_main_grn = MIN_GREEN_TIME;
      leftturn[direction  ].event = false;
      leftturn[direction+2].event = false;
      // left turn green if traffic is waiting in lane
      if (leftturn[direction].traffic_sensor_bit >= 0)
        if ((sensors & MASK(leftturn[direction].traffic_sensor_bit)) == 0) 
        { leftturn[direction].color     = _GRN;
          leftturn[direction].start_sec = cycle_sec;
          leftturn[direction].end_sec   = cycle_sec + MIN_GREEN_TIME; 
        }
      if (leftturn[direction+2].traffic_sensor_bit >= 0)
        if ((sensors & MASK(leftturn[direction+2].traffic_sensor_bit)) == 0)
        { leftturn[direction+2].color     = _GRN;
          leftturn[direction+2].start_sec = cycle_sec;
          leftturn[direction+2].end_sec   = cycle_sec + MIN_GREEN_TIME; 
        }
    }
    else // pedestrian crossing mode, 30 secs min green
    { min_main_grn = PEDESTRIAN_CROSSING_TIME;
      walk_signal[(direction^1)  ].count = PEDESTRIAN_CROSSING_TIME;
      walk_signal[(direction^1)+2].count = PEDESTRIAN_CROSSING_TIME;
VBUG(Serial.println(F("Pedestrian crossing mode"));)
    }
    // main green on if no opposing left turn is on
    if (leftturn[direction+2].color == _RED)
    { main_grn[direction].color     = _GRN;
      main_grn[direction].start_sec = cycle_sec;
      main_grn[direction].end_sec   = cycle_sec + min_main_grn;
    }
    if (leftturn[direction].color == _RED)
    { main_grn[direction+2].color     = _GRN;
      main_grn[direction+2].start_sec = cycle_sec;
      main_grn[direction+2].end_sec   = cycle_sec + min_main_grn;
    }
VBUG(Serial.println(F("Green lights activated"));)

    // wait for all greens to end
    for (uint8_t i=0; i < 4; i++) leftturn[i].event = false;
    do
    { coYield;

      for (uint8_t i=0; i < 4; i++)
        if (leftturn[i].event)
        { leftturn[i].event = false;
          // check for everything red or near-red (race condition)
          boolean is_red = true;
          for (uint8_t i=0; i < 4; i++)
            if (main_grn[i].color != _RED || main_grn[i].end_sec > cycle_sec+6
             || leftturn[i].color != _RED || leftturn[i].end_sec > cycle_sec+6)
             { is_red = false; break; }
          if (!is_red)
          { main_grn[(i+2)%4].color = _GRN; // release opposing traffic
            main_grn[(i+2)%4].start_sec = cycle_sec;
            main_grn[(i+2)%4].end_sec   = cycle_sec + MIN_GREEN_TIME;
          }
        }
    }      
    while (main_grn[direction].color != _RED || main_grn[direction+2].color != _RED
       ||  leftturn[direction].color != _RED || leftturn[direction+2].color != _RED);
VBUG(Serial.println(F("All lights red"));)

    // run left turn after pedestrians
    if (ped_mode)
    { if (leftturn[direction].traffic_sensor_bit >= 0)
        if ((sensors & MASK(leftturn[direction].traffic_sensor_bit)) == 0)
        { leftturn[direction].color     = _GRN;
          leftturn[direction].start_sec = cycle_sec;
          leftturn[direction].end_sec   = cycle_sec + MIN_GREEN_TIME; 
        }
      if (leftturn[direction+2].traffic_sensor_bit >= 0)
        if ((sensors & MASK(leftturn[direction+2].traffic_sensor_bit)) == 0)
        { leftturn[direction+2].color     = _GRN;
          leftturn[direction+2].start_sec = cycle_sec;
          leftturn[direction+2].end_sec   = cycle_sec + MIN_GREEN_TIME; 
        }
      // wait for all greens to end
VBUG(Serial.println(F("Running post-ped left turn")); )
     do
      { coYield;
      } while (leftturn[direction].color != _RED || leftturn[direction+2].color != _RED);
    }   
VBUG(Serial.println(F("Cycle ended."));)

    // switch directions
    direction ^= 1;

  coEnd
}

//-------------------------------------------------------------------

void monitorSensorsTask() // monitor sensors, continuously recalculate green times
{
  // sample & hold pedestrian buttons
  if ((sensors & MASK(NS_PED_BIT)) == 0) ped_request_pending[0] = true;
  if ((sensors & MASK(EW_PED_BIT)) == 0) ped_request_pending[1] = true;

  uint16_t max_green = MAX_GREEN_TIME;
  
  // pedestrian request shortens green
  if (ped_request_pending[direction]) max_green = CROSSTRAFFIC_GREEN_MAX_TIME;
  
  // opposing traffic shortens green
  for (uint8_t i=0; i < 4; i++)
  {
    // left turn opposing is oncoming + left/right
    if (leftturn[i].color == _GRN)
      if ((sensors & MASK(main_grn[(i+2)%4].traffic_sensor_bit)) == 0 
       || (sensors & MASK(main_grn[(i+1)%4].traffic_sensor_bit)) == 0
       || (sensors & MASK(main_grn[(i+3)%4].traffic_sensor_bit)) == 0)
        max_green = min( max_green, CROSSTRAFFIC_GREEN_MAX_TIME );
       
    // thru opposing is left/right + oncoming left turn
    if (main_grn[i].color == _GRN)
    { boolean lt = false;
      if (leftturn[(i+2)%4].traffic_sensor_bit != -1) lt = (sensors & MASK(leftturn[(i+2)%4].traffic_sensor_bit)) == 0;
      if ((sensors & MASK(main_grn[(i+1)%4].traffic_sensor_bit)) == 0 
       || (sensors & MASK(main_grn[(i+3)%4].traffic_sensor_bit)) == 0
       || lt)
        max_green = min( max_green, CROSSTRAFFIC_GREEN_MAX_TIME );
    }
  }
     
  // extend time while traffic is in lane
  for (uint8_t i=0; i < 4; i++)
  {
    if (leftturn[i].color == _GRN)
      if ((sensors & MASK(leftturn[i].traffic_sensor_bit)) == 0) 
      { // extend left turn time
        uint16_t max_sec = leftturn[i].start_sec + 60;// woof
        leftturn[i].end_sec = min( max_sec, cycle_sec+GREEN_EXTEND_TIME );
      }
  
    if (main_grn[i].color == _GRN)
      if ((sensors & MASK(main_grn[(i+2)%4].traffic_sensor_bit)) == 0) // got traffic?
      { // if opposing traffic limit green time 
        if ((sensors & MASK(main_grn[(i+1)%4].traffic_sensor_bit)) == 0
         || (sensors & MASK(main_grn[(i+3)%4].traffic_sensor_bit)) == 0)
          max_green = min( max_green, CROSSTRAFFIC_GREEN_MAX_TIME );
        // extend green time
        uint16_t max_sec = main_grn[i].start_sec + max_green;
        main_grn[i].end_sec = min( max_sec, cycle_sec+GREEN_EXTEND_TIME );
      }
  }

  // align green end times
  uint16_t best_end_sec = 0;
  for (uint8_t i=0; i < 4; i++)
  { if (main_grn[i].color == _GRN)
      if (best_end_sec < main_grn[i].end_sec) best_end_sec = main_grn[i].end_sec; 
    if (leftturn[i].color == _GRN)
      if (best_end_sec < leftturn[i].end_sec) best_end_sec = leftturn[i].end_sec; 
  }   
  for (uint8_t i=0; i < 4; i++)
  { // align thru lights
    if (main_grn[i].color == _GRN)
      main_grn[i].end_sec = best_end_sec;
    // align left turn
    if (leftturn[i].color == _GRN)
      if (leftturn[i].end_sec > best_end_sec) leftturn[i].end_sec = best_end_sec;
  }
}

//-------------------------------------------------------------------

void outputLightsTask() // display stoplight_color[] & leftturn_color[] on stoplight hardware (LED strings)
{ 
  static light_state last_color[4] = { _OFF, _OFF, _OFF, _OFF };
  static light_state turn_color[4] = { _OFF, _OFF, _OFF, _OFF };
  
  boolean update_left = false;
  boolean update_main = false;
  for (byte i=0; i < 4; i++)
  { byte x = LAMP_MAP[i];
    update_main |= (last_color[i] != main_grn[i].color);
    update_left |= (turn_color[i] != leftturn[i].color);
    if (last_color[i] != main_grn[i].color || turn_color[i] != leftturn[i].color)
    { last_color[i] = main_grn[i].color;
      turn_color[i] = leftturn[i].color;
      stoplight[x].setPixelColor( 0, main_grn[i].color == _RED ? stoplight[i].Color(STOPLIGHT_BRIGHTNESS,  0,0):stoplight[i].Color(0,0,0) );
      stoplight[x].setPixelColor( 1, main_grn[i].color == _YEL ? stoplight[i].Color(STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS,0):stoplight[i].Color(0,0,0) );
      stoplight[x].setPixelColor( 2, main_grn[i].color == _GRN ? stoplight[i].Color(0,STOPLIGHT_BRIGHTNESS,0):stoplight[i].Color(0,0,0) );
      if      (leftturn[i].color == _GRN) stoplight[x].setPixelColor( 3, stoplight[i].Color(0,STOPLIGHT_BRIGHTNESS,0) );
      else if (leftturn[i].color == _YEL) stoplight[x].setPixelColor( 3, stoplight[i].Color(STOPLIGHT_BRIGHTNESS,STOPLIGHT_BRIGHTNESS,0) );
      else if (leftturn[i].color == _RED) stoplight[x].setPixelColor( 3, stoplight[i].Color(STOPLIGHT_BRIGHTNESS,0,0) );
      else                                stoplight[x].setPixelColor( 3, stoplight[i].Color(0,0,0) );
      stoplight[x].show();
VBUG(Serial.print("Light "); Serial.print(i); Serial.print(" "); Serial.print( COLOR_NAME[main_grn[i].color] ); Serial.print("/"); Serial.println( COLOR_NAME[leftturn[i].color] );)
    }
  }
  // send to Processing
#ifdef processing
  if (update_main) { Serial1.write('A'); Serial1.write(main_grn[0].color + '0'); for (uint8_t i=1; i < 4; i++) { Serial1.print(','); Serial1.write(main_grn[i].color + '0'); } Serial1.println(); }
  if (update_left) { Serial1.write('B'); Serial1.write(leftturn[0].color + '0'); for (uint8_t i=1; i < 4; i++) { Serial1.print(','); Serial1.write(leftturn[i].color + '0'); } Serial1.println(); }
#endif
}

//-------------------------------------------------------------------

void doPedestrianLight( uint8_t light_index )
{
#ifdef HAZIO
  if (walk_display[walk_signal[light_index].pole1].uint_2digit( walk_signal[light_index].unit1, walk_signal[light_index].count )) fault(E_HT16K33);
  if (walk_display[walk_signal[light_index].pole2].uint_2digit( walk_signal[light_index].unit2, walk_signal[light_index].count )) fault(E_HT16K33);
#endif
VBUG(Serial.print("Walk "); Serial.print(light_index); Serial.print(" "); Serial.println(walk_signal[light_index].count);)
#ifdef processing
  Serial1.print('D'); Serial1.print(walk_signal[light_index].pole1); Serial1.print(','); Serial1.print(walk_signal[light_index].unit1); Serial1.print(','); Serial1.println(walk_signal[light_index].count);
  Serial1.print('D'); Serial1.print(walk_signal[light_index].pole2); Serial1.print(','); Serial1.print(walk_signal[light_index].unit2); Serial1.print(','); Serial1.println(walk_signal[light_index].count);
#endif
}

void NpedestrianLight()
{
  if (walk_signal[N_LIGHT].count == 0) return;
  coBegin
    for (; walk_signal[N_LIGHT].count; walk_signal[N_LIGHT].count--)
    { doPedestrianLight(N_LIGHT);
      coDelayWarped(1000);
    }
    doPedestrianLight(N_LIGHT);
  coEnd
}

void EpedestrianLight()
{
  if (walk_signal[E_LIGHT].count == 0) return;
  coBegin
    for (; walk_signal[E_LIGHT].count; walk_signal[E_LIGHT].count--)
    { doPedestrianLight(E_LIGHT);
      coDelayWarped(1000);
    }
    doPedestrianLight(E_LIGHT);
  coEnd
}

void SpedestrianLight()
{
  if (walk_signal[S_LIGHT].count == 0) return;
  coBegin
    for (; walk_signal[S_LIGHT].count; walk_signal[S_LIGHT].count--)
    { doPedestrianLight(S_LIGHT);
      coDelayWarped(1000);
    }
    doPedestrianLight(S_LIGHT);
  coEnd
}

void WpedestrianLight()
{
  if (walk_signal[W_LIGHT].count == 0) return;
  coBegin
    for (; walk_signal[W_LIGHT].count; walk_signal[W_LIGHT].count--)
    { doPedestrianLight(W_LIGHT);
      coDelayWarped(1000);
    }
    doPedestrianLight(W_LIGHT);
  coEnd
}
