//------------------------------------------------------------
// Dual 8 digit 7-segment display driver (HT16K33)
// M405 PCB
// 2022.04.28 RSP
//------------------------------------------------------------

#include <Wire1.h>
#define MYWIRE Wire1

// I2C device found at address 0x70  !
#define HT16K33_DEFAULT_I2C_ADDR 0x70 // default module address

#define HT16K33_BLINK_CMD       0x80
#define HT16K33_BLINK_DISPLAYON 0x01
#define HT16K33_BLINK_OFF     0
#define HT16K33_BLINK_2HZ     1
#define HT16K33_BLINK_1HZ     2
#define HT16K33_BLINK_HALFHZ  3
#define HT16K33_CMD_BRIGHTNESS  0xE0

const uint8_t SEG_TABLE[] = // 7 segment characters
{// .gfedcba
  0b00111111, // 0
  0b00000110, // 1
  0b01011011, // 2
  0b01001111, // 3
  0b01100110, // 4
  0b01101101, // 5
  0b01111101, // 6
  0b00000111, // 7
  0b01111111, // 8
  0b01101111, // 9
  0b01110111, // A
  0b01111100, // b 
  0b00111001, // C
  0b01011110, // d
  0b01111001, // E
  0b01110001, // F
  0b00000000, // blank
  0b01000000, // dash
  0b01010100  // wtf
};
#define CH_BLANK 16
#define CH_DASH  17
#define CH_WTF   18
uint8_t segments[8]; // display buffer

class stoplight_7seg_HT16K33
{
  private:
    uint8_t i2c_address; // this module

    uint8_t send( uint8_t v ) // helper to send a command to HT16K33
    { MYWIRE.beginTransmission(i2c_address);
        MYWIRE.write(v);
      return MYWIRE.endTransmission();  
    }

  public:
    stoplight_7seg_HT16K33( const uint8_t addr=HT16K33_DEFAULT_I2C_ADDR ) { i2c_address = addr; }
    uint8_t set_brightness( uint8_t b ) { return send(HT16K33_CMD_BRIGHTNESS | b); }
    uint8_t set_display( boolean isOn ) { return send(HT16K33_BLINK_CMD | (isOn ? HT16K33_BLINK_DISPLAYON : 0) ); }
    uint8_t turnon_display()            { return send(0x21); } 
    
    uint8_t begin(uint8_t brightness) 
    { uint8_t e;
      if ((e = turnon_display())) return e;
      if ((e = set_display(true))) return e;
      if ((e = set_brightness(brightness))) return e;
      return 0; // success
    }

    uint8_t uint_2digit( uint8_t unit, uint8_t value )
    {
      if (value > 99) return 99; // failsafe
      byte tens,ones;
      tens = value / 10;
      ones = value % 10;
      MYWIRE.beginTransmission(i2c_address);
        MYWIRE.write(unit*4); // 2 digits per unit, 2 bytes per digit
        uint16_t segs = SEG_TABLE[ value > 9 ? tens:16] << 2; // blank leading zero
        if (value <= 10) segs |= 0x0400; // red LED on
        MYWIRE.write( segs & 0xFF );
        MYWIRE.write( segs >> 8 );
        segs = SEG_TABLE[ones] << 2;   
        if (value > 10) segs |= 0x0400; // green LED on
        MYWIRE.write( segs & 0xFF );
        MYWIRE.write( segs >> 8 );
      return MYWIRE.endTransmission();
    }
};
