//-------------------------------------------------------------------
// Stoplight Simulator display and interactive interface
// 2024.04.30 RSP
//
// Cars are clickable to simulate traffic presence
// Pedestrian buttons are clickable
//    Red = not pressed
//    Blu = pending
//    Grn = crossing
// Speed buttons are clickable for accellerated simulation
//-------------------------------------------------------------------

import processing.serial.*;

Serial myPort; // RP2040 stoplight controller

PImage img; // street intersection background image
int stoplight_radius = 15;
int speed_scale = 0; // x1

// receive commands from stoplight controller
int rxBP;
char[] rxBuf = new char[10];
int[] csv = new int[10];

// simulated cars
int[] car_x = new int[6];
int[] car_y = new int[6];
int[] car_orientation = new int[6];
int[] car_sensor_bit = new int[6];
boolean[] car_active = new boolean[6];

// pedestrian buttons
int[] btn_x = new int[8];
int[] btn_y = new int[8];
boolean[] btn_active = new boolean[8];
int[] btn_group = new int[8];
int ped_sensor_1_bit = 0;
int ped_sensor_2_bit = 1;

// walk time displays
int[] walk_x = new int[8];
int[] walk_y = new int[8];
int[] walk_count = new int[8]; // reflected

// stoplight displays
int[] light_x = new int[4];
int[] light_y = new int[4];
int[] light_color = new int[4];
int[] left_x = new int[4];
int[] left_y = new int[4];
int[] left_color = new int[4];

// walk display lookup table
int[][] walkLUT = { {0,1}, {2,3}, {4,5}, {6,7} };

void setup() 
{
  
// cars & pedestrian buttons are numbered counter-clockwise starting at west
// stoplights are numbered clockwise starting at north

  car_x[0]=345; car_y[0]=422; car_orientation[0]=0; car_sensor_bit[0]=2;
  car_x[1]=345; car_y[1]=477; car_orientation[1]=0; car_sensor_bit[1]=3;
  car_x[2]=584; car_y[2]=640; car_orientation[2]=1; car_sensor_bit[2]=5;
  car_x[3]=753; car_y[3]=423; car_orientation[3]=0; car_sensor_bit[3]=7;
  car_x[4]=753; car_y[4]=373; car_orientation[4]=0; car_sensor_bit[4]=6;
  car_x[5]=518; car_y[5]=205; car_orientation[5]=1; car_sensor_bit[5]=4;

  walk_x[0]=370; walk_y[0]=530;
  walk_x[1]=435; walk_y[1]=590;
  walk_x[2]=640; walk_y[2]=590;
  walk_x[3]=700; walk_y[3]=530;
  walk_x[4]=700; walk_y[4]=290;
  walk_x[5]=640; walk_y[5]=230;
  walk_x[6]=435; walk_y[6]=230;
  walk_x[7]=370; walk_y[7]=290;
  
  btn_x[0]=415; btn_y[0]=545; btn_group[0]=0;
  btn_x[1]=450; btn_y[1]=577; btn_group[1]=1;
  btn_x[2]=655; btn_y[2]=577; btn_group[2]=1;
  btn_x[3]=686; btn_y[3]=545; btn_group[3]=0;
  btn_x[4]=686; btn_y[4]=303; btn_group[4]=0;
  btn_x[5]=655; btn_y[5]=273; btn_group[5]=1;
  btn_x[6]=450; btn_y[6]=273; btn_group[6]=1;
  btn_x[7]=415; btn_y[7]=303; btn_group[7]=0;
 
  light_x[3]=470; light_y[3]=465; // W
  light_x[2]=586; light_y[2]=510; // S
  light_x[1]=640; light_y[1]=360; // E
  light_x[0]=517; light_y[0]=310; // N
  
  left_x[1]=470; left_y[1]=425; // E Eastbound->Northbound
  left_x[3]=640; left_y[3]=420; // W Westbound->Southbound

  // intersection image
  size(1115,846);
  img = loadImage("5b.jpg");
  
  // List all the available serial ports
  printArray(Serial.list());
  // if wrong port, change index to correct port here...
  String portName = Serial.list()[1];
  myPort = new Serial(this, portName, 115200);
}

//-------------------------------------------------------------------

// draw left turn signal stoplight
void drawLeft(int x, int y, int zcolor)
{
  switch (zcolor)
  { case 0: fill(255,0,0); break;
    case 2: fill(255,255,0); break;
    case 1: fill(0,255,0); break;
  }
  circle( x,y,stoplight_radius);
}
// draw thru stoplight
void drawLight(int x, int y, int zcolor)
{
  if (zcolor == 0) fill(255,0,0); else fill(204);
  circle( x,y,stoplight_radius);
  if (zcolor == 2) fill(255,255,0); else fill(204);
  circle( x,y+stoplight_radius,stoplight_radius);
  if (zcolor == 1) fill(0,255,0); else fill(204);
  circle( x,y+2*stoplight_radius,stoplight_radius);
}
// draw walk counter
void drawWalk(int x,int y, int count)
{
  fill(0,0,0);
  rect(x,y,30,30);
  if (count > 0) fill(0,255,0); else fill( 255,0,0 );
  textSize(25);
  textAlign(RIGHT);
  text( count, x+28,y+23 );     
}
// draw pedestrian button
void drawButton(int x,int y,boolean active,boolean running)
{
  if (running) fill(0,255,0); else
    if (active) fill(0,0,255); else fill(0,0,0);
  circle(x,y,20);
  if (active) fill(0,0,0); else fill(255,0,0);
  circle(x,y,10);
}

//-------------------------------------------------------------------

// send sensor (car & pedestrian button) state to stoplight controller
void sendSensors( byte ped_bits )
{
  byte s = 0;
  for (int i=0; i < 6; i++) if (car_active[i]) s |= (1 << car_sensor_bit[i]);
  s |= ped_bits;
  myPort.write('S'); // ID for data byte that follows
  myPort.write(s);
}

//-------------------------------------------------------------------

// detect & handle mouse hits on buttons
void mousePressed()
{
  // toggle car presence
  for (int i=0; i < 6; i++)
  { int w = (car_orientation[i]==0) ? 50:25;
    int h = (car_orientation[i]==0) ? 25:50;
    if (mouseX >= car_x[i]-w && mouseX <= car_x[i]+w && mouseY >= car_y[i]-h && mouseY <= car_y[i]+h)
    { car_active[i] = !car_active[i];
      sendSensors((byte)0);
      return;
    }
  }
  // pedestrian buttons
  //   lights up all buttons in the group
  //   sends a button pulse to the stoplight controller
  { int r = 20;
    for (int i=0; i < 8; i++)
      if (mouseX >= btn_x[i]-r && mouseX <= btn_x[i]+r && mouseY >= btn_y[i]-r && mouseY <= btn_y[i]+r)
      { btn_active[i] = true;
        // light up all buttons in group
        for (int j=0; j < 8; j++)
          if (btn_group[i] == btn_group[j]) btn_active[j] = true;
        // pulse button to simulate momentary button press
        byte b = 0;
        if (btn_active[0]) b |= (1 << ped_sensor_2_bit);
        if (btn_active[1]) b |= (1 << ped_sensor_1_bit);
        sendSensors(b);
        sendSensors((byte)0);
        return;
      } 
  }
  // speed select buttons
  for (int i=0; i < 3; i++)
  { int x = 25;
    int y = 50+40*i;
    int w = 50;
    int h = 40;
    if (mouseX >= x && mouseX <= x+w && mouseY >= y && mouseY <= y+h)
    { speed_scale = i;
      myPort.write('X'); // ID for data byte that follows
      myPort.write((byte)speed_scale);
    }
  }
}

//-------------------------------------------------------------------

void draw()
{
  
//-- process serial data from stoplight controller

  while( myPort.available() > 0 ) 
  {
    int c = myPort.read();
    if (c >= 'A' && c <= 'D')
    { rxBuf[0] = (char)c;
      rxBP = 1;
    }
    else if (c == 10)
      ; // ignore LF
    else if (c == 13)
    { // parse input (ASCII CSV to csv[])
//for (int i=0; i < rxBP; i++) print(rxBuf[i]); println();
      int csv_count = 0;
      for (int i=1; i < rxBP; i++)
        if (rxBuf[i] >= '0' && rxBuf[i] <= '9')
        { csv[csv_count] = rxBuf[i]-'0';
          csv_count++;
          for (i++; i < rxBP; i++)
            if (rxBuf[i] >= '0' && rxBuf[i] <= '9')
              csv[csv_count-1] = csv[csv_count-1]*10 + rxBuf[i]-'0';
            else break;
        }
//print("CSV ("); print(csv_count); print(")"); for (int i=0; i < csv_count; i++) { print(" "); print(csv[i]); } println();
      // handle data type
      switch (rxBuf[0])
      {
        case 'A': // main grn states
          if (csv_count == 4)
            for (int i=0; i < 4; i++)
              light_color[i] = csv[i];
          break;
        case 'B': // left turn states
          if (csv_count == 4)
            for (int i=0; i < 4; i++)
              left_color[i] = csv[i];
          break;
        case 'D': // walk countdown
          if (csv_count == 3)
            walk_count[walkLUT[csv[0]][csv[1]]] = csv[2];
          break;
        case 'C': // clear ped cycle
          if (csv_count == 1)
            for (int i=0; i < 8; i++)
              if (csv[0] == btn_group[i]) btn_active[i] = false;
      }
    }
    else if (rxBP < 10) rxBuf[rxBP++] = (char)c;
  }

//-- render the display

  image(img, 0, 0);
  
  fill( 0,0,0 );
  textSize(25);
  textAlign(LEFT);
  text("Stoplight Simulator", 25,30 );
  
//-- speed buttons

  for (int i=0; i < 3; i++)
  {
    if (i == speed_scale)
         fill( 0,255,0 );
    else fill( 192,192,192 );
    rect( 25,50+40*i,50,40);
    if (i == speed_scale)
          fill( 0,0,0 );
    else  fill( 0,0,0 );
    textSize(25);
    textAlign(LEFT);
    text( i==0 ? "X1" : i==1 ? "X2" : "X4", 39,77+i*40 );      
  }

//-- dynamic elements

  for (int i=0; i < 4; i++)
    drawLight( light_x[i], light_y[i], light_color[i] );
  for (int i=1; i < 4; i+=2)
    drawLeft( left_x[i], left_y[i], left_color[i] );
 
  fill(60,255,64);
  for (int i=0; i < 6; i++)
    if (car_active[i])
      ellipse(car_x[i],car_y[i],(car_orientation[i]!=0) ? 20:50,(car_orientation[i]!=0) ? 50:20 );
      
  for (int i=0; i < 8; i++)
  { drawWalk(walk_x[i],walk_y[i], walk_count[i]);
    drawButton(btn_x[i],btn_y[i],btn_active[i],walk_count[i] > 0);
  }
 
}
