//------------------------------------------------------
// Touch Alarm Clock 2024
// 2024.07.20 RSP
// Target: RP2040
//         DFplayer
//         ILI9341 + XPT2046 touch
//------------------------------------------------------

#include "app.h"        // constants

//#define DEBUG_CONSOLE   // enable debug output

//-------------------------------------------------------------------
// DFplayer (knock-off from AE) 
//   MP3-TF-16P

#include <DFPlayerMini_Fast.h> // https://github.com/PowerBroker2/DFPlayerMini_Fast
DFPlayerMini_Fast myDFPlayer;

//------------------------------------------------------
// 240x320 Display (ILI9341+XPT2046)

// using Setup_Fearless_TeensySynth.h
#include "User_Setup_Select.h"
#ifndef Setup_Fearless_TeensySynth
#error "WRONG SCREEN DRIVER"
#endif
#include "Free_Fonts.h" // Include the header file attached to this sketch
#include "SPI.h"
#include "TFT_eSPI.h"
TFT_eSPI tft = TFT_eSPI(); // Using hardware SPI

#include <XPT2046_Touchscreen.h>
#define CS_PIN  21 // T_CS
XPT2046_Touchscreen ts(CS_PIN);

#include "pages.h"
PageController page_controller;

void brightnessTask()
{ static int _state_ = 0;  static uint32_t _tm_;
  static uint32_t total = 0; // running average
  static uint     count;
  coBegin
    for (count=0; count < 120; count++) // sample for 30 seconds 
    { coDelay(250)
      total -= total/120;
      total += analogRead( BRIGHTNESS_PIN );
    }
    total /= 120; // calculate average
    total = constrain( total, 25, 250 );
    total = map( total, 25,250, 64,255 );
    analogWrite( BL_PIN, total);
    count = 0;
  coEnd
}

void dumpCalibration()
{
#ifdef DEBUG_CONSOLE
  Serial.print("Touch calibration: "); Serial.print(page_controller.touch_cal.min_x);
  Serial.print(","); Serial.print(page_controller.touch_cal.max_x);
  Serial.print(","); Serial.print(page_controller.touch_cal.min_y);
  Serial.print(","); Serial.println(page_controller.touch_cal.max_y);
#endif
}

//-------------------------------------------------------------------
// error handler
//   blinks error code on the snooze & alarm LEDs

#define BLINK_DELAY_MS 1000
#define BLINK_ON_MS     100
#define BLINK_OFF_MS    500

uint fault_code =  0;
#define E_DS3231   1 // I2C bus error
#define E_HT16K33  2 // I2C bus error
#define E_DFPLAYER 3 // DFplayer failed to start
#define E_RTC      4 // RTC not providing updates (1Hz SQW)

void fault( uint8_t blinks ) // set fault condition
{
  fault_code = blinks;
#ifdef DEBUG_CONSOLE  
Serial.print(F("FAULT: ")); Serial.println(blinks);
#endif
}
void faultTask() // blink error code in real time
{ static uint8_t count;
  static int _state_ = 0;
  static ulong _tm_;
  coBegin
    if (fault_code)
    { count = fault_code;
      digitalWrite( POWER_PIN, LOW );
      coDelayWhile( BLINK_DELAY_MS, fault_code != 0 );
      for (; count; count--)
      { digitalWrite( POWER_PIN, HIGH );
        coDelayWhile( BLINK_ON_MS, fault_code != 0 );
        digitalWrite( POWER_PIN, LOW );
        coDelayWhile( BLINK_OFF_MS, fault_code != 0 );
      }
    }
  coEnd
}

//-------------------------------------------------------------------
// DS3231M RTC
//   time is in Hours:Minutes
//   alarm is stored in A1_Hours:A1:Minutes

#include "Wire.h"

#define DS3231_DEFAULT_I2C_ADDR 0x68

boolean rtc_update_available = false;
boolean simulate_rtc_failure = false;

void SQW_interrupt()
{
  rtc_update_available = true;
}

enum DS3231_REGS : uint8_t
{ 
  SECONDS = 0x00,
  MINUTES,
  HOURS,
  DAY_OF_WEEK,
  DATE,
  MONTH_CENTURY,
  YEAR,
  A1_SECONDS,
  A1_MINUTES,
  A1_HOURS,
  A1_DAY_DATE,
  A2_MINUTES,
  A2_HOURS,
  A2_DAY_DATE,
  CONTROL,
  STATUS,
  AGING_OFFSET,
  MSB_TEMP,
  LSB_TEMP
};

void DS3231write( byte reg, byte data )
{
  Wire.beginTransmission(DS3231_DEFAULT_I2C_ADDR);
    Wire.write((uint8_t)reg);
    Wire.write((uint8_t)data);
  if (Wire.endTransmission()) fault(E_DS3231);
}
int DS3231read( byte reg )
{
  Wire.beginTransmission(DS3231_DEFAULT_I2C_ADDR);
    Wire.write((uint8_t)reg);
  if (Wire.endTransmission()) { fault(E_DS3231); return -1; }
  Wire.requestFrom(DS3231_DEFAULT_I2C_ADDR, (uint8_t)1 );
  if (!Wire.available()) { fault(E_DS3231); return -1; }
  return Wire.read();
}
void DS3231get( byte reg, struct ClockTime *t ) // reg == DS3231_REGS::SECONDS or DS3231_REGS::A1_SECONDS
{
  Wire.beginTransmission(DS3231_DEFAULT_I2C_ADDR);
    Wire.write(reg);
  if (Wire.endTransmission()) { fault(E_DS3231); return; }
  Wire.requestFrom(DS3231_DEFAULT_I2C_ADDR, (uint8_t)3 );
  if (!Wire.available()) { fault(E_DS3231); return; } else t->seconds = Wire.read();
  if (!Wire.available()) { fault(E_DS3231); return; } else t->minutes = Wire.read();
  if (!Wire.available()) { fault(E_DS3231); return; } else t->hours   = Wire.read();
  if ((t->hours & 0x1F) == 0) { t->hours += 1; DS3231put( reg, t ); } // fix invalid 12 hour value
  t->pm = (t->hours & 0x20) != 0;
  t->hours &= 0x1F;
}
void DS3231put( byte reg, struct ClockTime *t ) // reg == DS3231_REGS::SECONDS or DS3231_REGS::A1_SECONDS
{
  Wire.beginTransmission(DS3231_DEFAULT_I2C_ADDR);
    Wire.write(reg);
    Wire.write(t->seconds);
    Wire.write(t->minutes);
    Wire.write(t->hours + 0x40 + (t->pm ? 0x20:0) );
  if (Wire.endTransmission()) { fault(E_DS3231); return; }
} 

#ifdef DEBUG_CONSOLE  
void dumpDS3231()
{
  Serial.println("DS3231 Register Dump");
  Wire.beginTransmission(DS3231_DEFAULT_I2C_ADDR);
    Wire.write(0);
  if (Wire.endTransmission()) { fault(E_DS3231); return; }
  Wire.requestFrom(DS3231_DEFAULT_I2C_ADDR, (uint8_t)17 );
  for (uint i=0; Wire.available(); i++)
  { Serial.print(i); Serial.print(": 0x"); Serial.println(Wire.read(),HEX); }
}
#endif

//------------------------------------------------------

class coRTC
{
  private:
    int _state_ = 0; // coroutine state
    uint32_t _tm_;   // for delay functions

  public:
    struct ClockTime t = {1,0,0,false}; // the clock time
    uint los_count = 0;
    
    void loop()
    { coBegin
        coDelayWhile( 1000, rtc_update_available == false )
if (simulate_rtc_failure) rtc_update_available = false;      
        if (rtc_update_available)
        { rtc_update_available = false;
          // read RTC registers
          DS3231get( DS3231_REGS::SECONDS, &t );
          los_count = 0;
        }
        else // RTC timed out
          if (++los_count >= 10)
          { fault( E_RTC );
            los_count = 0;
          }
      coEnd
    }
};
coRTC rtcTask;

//------------------------------------------------------
// helper callbacks from time/alarm setting pages

// get the time
void getTime( struct ClockTime *t )
{
  memcpy( t, &rtcTask.t, sizeof(struct ClockTime) );
}

// set the time
void setTime( struct ClockTime *t )
{
  t->seconds = 0; // sync seconds  
  memcpy( &rtcTask.t, t, sizeof(struct ClockTime) );
  DS3231put( DS3231_REGS::SECONDS, t );
}

// set the alarm
void setAlarm( struct ClockTime *t )
{
  memcpy( &page_controller.clock_page.tAlarm, t, sizeof(struct ClockTime) );
  DS3231put( DS3231_REGS::A1_SECONDS, t );
}

//------------------------------------------------------
// Processor Core 0 = User Interface
//------------------------------------------------------

void setup() 
{
#ifdef DEBUG_CONSOLE  
  // diagnostics
  Serial.begin(115200);
  while (!Serial && millis() < 10000UL);
  Serial.println("core 0 start"); 
#endif

  // alarm LED
  pinMode( LIGHT_PIN, OUTPUT );  digitalWrite( LIGHT_PIN, LOW );
  // power & fault indicator
  pinMode( POWER_PIN, OUTPUT );  digitalWrite( POWER_PIN, HIGH );

  // LCD display
  tft.begin();
  tft.setRotation(1); // landscape, UL is 0,0
  tft.fillScreen(TFT_BLACK);
  tft.setTextColor(TFT_WHITE, TFT_BLACK);
  analogWrite( BL_PIN, 255); // full brightness to start up
  // touch panel
  ts.begin();
  ts.setRotation(3);    // landscape, 0,0 = UL corner
  // load touch panel calibration
  EEPROM.begin(256);
  EEPROM.get(0,page_controller.touch_cal);

  // start I2C & verify devices are on the bus
  Wire.begin();
  Wire.beginTransmission(DS3231_DEFAULT_I2C_ADDR); if (Wire.endTransmission()) fault(E_DS3231);

  // set up RTC
  DS3231write( DS3231_REGS::CONTROL, 0x18 ); // set SQW to output 1Hz & enable oscillator
  DS3231write( DS3231_REGS::HOURS, DS3231read( DS3231_REGS::HOURS ) | 0x40 ); // set 12 hour mode
  // get current RTC values for time & alarm
  DS3231get( DS3231_REGS::A1_SECONDS, &page_controller.clock_page.tAlarm );
  DS3231get( DS3231_REGS::SECONDS,    &rtcTask.t );
  
  // 1Hz RTC (SQW)
  pinMode(SQW_PIN, INPUT_PULLUP);  
  attachInterrupt( digitalPinToInterrupt(SQW_PIN), SQW_interrupt, FALLING );

  // start up DFplayer
  Serial1.begin(9600);
  if (!myDFPlayer.begin(Serial1, false)) fault(E_DFPLAYER);
 
  // show first page
  page_controller.start(PG_SPLASH);
}

// - - - - - - - - - - - - -
// core 0 loop
//   can stall (screen updates often take > 1 second)

void loop(void) // UI, allowed to block as needed
{

//-- RTC is the timebase

  rtcTask.loop();

//-- alarm check

  if (page_controller.clock_page.alarm_enabled)   // alarm armed ?
    if (page_controller.current_page != PG_ALARM  // not already alarming ?
     && page_controller.current_page != PG_SNOOZE)
    { struct ClockTime t;
      getTime( &t );
      if (t.hours == page_controller.clock_page.tAlarm.hours && t.minutes == page_controller.clock_page.tAlarm.minutes)
      { page_controller.alarm_page.mode = 0;
        page_controller.start(PG_ALARM);
      }
    }
            
//-- drive the user interface

  page_controller.loop();

//-- blink the lamp if an error condition exists

  faultTask();

//-- automatic brightness

  brightnessTask();
  
//-- diagnostic console on serial monitor

#ifdef DEBUG_CONSOLE
  { int c = Serial.read();
    switch (c)
    { case -1 : break;

      case '1':
        myDFPlayer.volume(10);
        myDFPlayer.play(1);
        break;
      case '2':
        myDFPlayer.volume(10);
        myDFPlayer.play(2);
        break;
        
      case 'A': case 'a': // trigger alarm
        page_controller.start( PG_ALARM );
        break;

      case 'C': case 'c': // touch calibration
        dumpCalibration();
        break;
        
      case 'D': case 'd': // dump DS3231 registers
        dumpDS3231();
        break;

      case 'R': case 'r': // RTC failure simulation
        simulate_rtc_failure = !simulate_rtc_failure;
        Serial.print("RTC failure simulation: "); Serial.println(simulate_rtc_failure ? "ON":"OFF");
        break;
                
      case 'T': case 't': // calibrate touch
        page_controller.start( PG_CAL );
        break;        
    }
  }
#endif  
}
