//------------------------------------------------------
// screen displays

#include "app.h"
#include "pages.h"

// title font
#include "font_7seg.h" // http://oleddisplay.squix.ch/#/home
#define CF_DSEG7 &DSEG7_Classic_Bold_80

#include <DFPlayerMini_Fast.h> // https://github.com/PowerBroker2/DFPlayerMini_Fast

// in main...
extern PageController page_controller;
extern void getTime( struct ClockTime *t );
extern void setTime( struct ClockTime *t );
extern void setAlarm( struct ClockTime *t );

// display geometry
#define DISP_X 20
#define ROW1_Y 22
#define ROW2_Y (120+ROW1_Y)

#define SET_TIMEOUT_MS (2UL*60000UL) // automatically cancel setting mode after 2 minutes
#define SET_TIME_Y 50

#define FLASH_ON_MS  1000
#define FLASH_OFF_MS  500

//------------------------------------------------------
// Helpers

// flash the lamp
struct coFlash
{
  private:
    int _state_ = 0; // coroutine state
    uint32_t _tm_;   // for delay functions
    boolean enabled = false;
    
  public:
    void control( boolean onoff )
    {
      if (onoff)
      { _state_ = 0;
        enabled = true;
      }
      else 
      { enabled = false;
        digitalWrite( LIGHT_PIN, LOW );
      }        
    }
    void loop()
    { coBegin
        coWaitWhile( !enabled )
        for (;;) // run until stopped
        { coDelay( FLASH_ON_MS )
          digitalWrite( LIGHT_PIN, HIGH );
          coDelay( FLASH_OFF_MS )
          digitalWrite( LIGHT_PIN, LOW );
        }
      coEnd
    }
};
coFlash flashTask;

// paint time or alarm
void paintTime( uint16_t color, struct ClockTime *t, uint y )
{  
  // time
  tft.setTextDatum(TL_DATUM);
  tft.setFreeFont(CF_DSEG7);             
  uint th1 = tft.fontHeight();
  tft.setTextColor( color, BACKGND_COLOR ); // overstrike mode
  uint x = DISP_X+5;
  String s = "";
  if (t->hours >> 4) 
    s = String(t->hours >> 4); 
  else
  { uint tw = tft.textWidth("8")+8;
    tft.fillRect( x,y, tw,tft.fontHeight(), BACKGND_COLOR );
    x += tw;
  }
  s = s + String(t->hours & 0x0F);
  s = s + ":";
  s = s + String(t->minutes >> 4) + String(t->minutes & 0x0F);
  tft.drawString( s, x, y );
  // AM/PM
  tft.setFreeFont(FSS12);             
  uint th2 = tft.fontHeight();
  tft.drawString( t->pm ? "PM":"AM", DISP_X, (th1-th2)/2 + y );
}

// create the buttons for time/alarm seting
void createSettingButtons(uint y)
{
  page_controller.createAreaButton(   0,  0,  160,y+40  ); // button 0 HH up
  page_controller.createAreaButton( 160,  0,   80,y+40  ); // button 1 M_ up
  page_controller.createAreaButton( 240,  0,   80,y+40  ); // button 2 _M up
  page_controller.createAreaButton(   0,y+50, 160,110-y ); // button 3 HH down
  page_controller.createAreaButton( 160,y+50,  80,110-y ); // button 4 M_ down
  page_controller.createAreaButton( 240,y+50,  80,110-y ); // button 5 _M down
}  

// handle time/alarm setting press
void adjustSetTime( struct ClockTime *tSet, int btn, uint16_t color )
{
  switch (btn)
  {
    case 0: // HH up
      tSet->hours += 0x01;
      if ((tSet->hours & 0x0F) > 0x09)   tSet->hours = 0x10;
      if ((tSet->hours       ) > 0x12) { tSet->hours = 0x01;  tSet->pm = !tSet->pm; }
      paintTime( color, tSet, SET_TIME_Y );
      break;
    case 1: // M_ up
      tSet->minutes += 0x10;
      if (tSet->minutes > 0x50) tSet->minutes = (tSet->minutes & 0x0F);
      paintTime( color, tSet, SET_TIME_Y );
      break;
    case 2: // _M up
      tSet->minutes += 0x01;
      if ((tSet->minutes & 0x0F) > 0x09) tSet->minutes = (tSet->minutes & 0xF0);
      paintTime( color, tSet, SET_TIME_Y );
      break;
    case 3: // HH down
      if (tSet->hours == 0x01) { tSet->hours = 0x12;  tSet->pm = !tSet->pm; }
      else if (tSet->hours == 0x10) tSet->hours = 0x09;
      else tSet->hours--;
      paintTime( color, tSet, SET_TIME_Y );
      break;
    case 4: // M_ down
      if ((tSet->minutes & 0xF0) == 0) tSet->minutes = (tSet->minutes & 0x0F) + 0x60;
      tSet->minutes -= 0x10;
      paintTime( color, tSet, SET_TIME_Y );
      break;
    case 5: // _M down
      if ((tSet->minutes & 0x0F) == 0) tSet->minutes = (tSet->minutes & 0xF0) + 0x0A;
      tSet->minutes -= 0x01;
      paintTime( color, tSet, SET_TIME_Y );
      break;
  }
}

// - - - - - - - - - - - - -
// Splash / Instructions page
// - - - - - - - - - - - - -

void splashPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSS18);
  tft.drawString( APP_NAME, DISP_W/2,100); 

  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setFreeFont(FSS12);             
  tft.drawString( "Touch to calibrate", DISP_W/2,200); 

  page_controller.clear(); // no buttons on this page
}

void splashPage::loop()
{
  coBegin
    coDelay( 5000 )
    page_controller.start(PG_CLOCK);
  coEnd
}

boolean splashPage::rawPress( [[maybe_unused]] TS_Point *pt )
{ // enter calibration if touched  
  page_controller.start(PG_CAL);
  return true; // press was handled
}

// - - - - - - - - - - - - -
// Clock page
// - - - - - - - - - - - - -

void clockPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  page_controller.clear(); // prepare to create buttons
  
  // paint time area
  last_t.hours = 99; // force refresh
  loop();
  page_controller.createAreaButton( 0,0, DISP_W,DISP_H/2 ); // button 0
  
  // paint alarm area
  if (alarm_enabled) // show alarm time
  { paintTime( TFT_YELLOW, &page_controller.clock_page.tAlarm, ROW2_Y );
    page_controller.createAreaButton( 0,DISP_H/2, DISP_W,DISP_H/2 ); // button 1
  }  
  else // show alarm options
  { tft.setFreeFont(FSS12);             
    tft.setTextColor( TFT_YELLOW, BACKGND_COLOR ); // overstrike mode
    tft.setTextDatum(TC_DATUM); 
    tft.drawString( "Alarm is off", DISP_W/2,ROW2_Y-20 );
    page_controller.createTextButton( 170,DISP_H-75, 100,70, "Set Alarm", FSS12 ); // button 1
    page_controller.createTextButton(  40,DISP_H-75, 100,70, "Arm Alarm", FSS12 ); // button 2
  }
}

void clockPage::loop()
{
  // only update if changed
  struct ClockTime t;
  getTime( &t );
  if (last_t.hours != t.hours || last_t.minutes != t.minutes)
  { last_t.hours   = t.hours;
    last_t.minutes = t.minutes;
    paintTime( TEXT_COLOR, &t, ROW1_Y );
  }
}

boolean clockPage::buttonPress( int btn )
{
  switch (btn)
  { case 0: // set time
      page_controller.start( PG_SETCLOCK );
      return true;
    case 1: // set alarm time
      page_controller.start( PG_SETALARM );
      return true;
    case 2: // turn alarm on
      alarm_enabled = true;
      init(); // update button configuration
      return true;
  }
  return false; // page was not changed
}

// - - - - - - - - - - - - -
// Clock Set page
//     HH:MM
//   set  cancel
// - - - - - - - - - - - - -

void clockSetPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  page_controller.clear(); // prepare to create buttons
  // grab a copy of time
  getTime( &tSet );  
  // paint time area
  paintTime( TEXT_COLOR, &tSet, SET_TIME_Y );
  // create buttons
  const uint BTN_Y = 190;
  page_controller.createTextButton(  50,BTN_Y,  100,45, "Set",    FSS12 ); // button 0
  page_controller.createTextButton( 170,BTN_Y,  100,45, "Cancel", FSS12 ); // button 1
  createSettingButtons(SET_TIME_Y);
}

void clockSetPage::loop()
{
  // time out setting (cancel) & return to normal clock mode
  coBegin
    coDelay( SET_TIMEOUT_MS )
  coEnd
  page_controller.start( PG_CLOCK );
}

boolean clockSetPage::buttonPress( int btn )
{
  _state_ = 0; // keep alive
  switch (btn)
  { 
    case 1: // Cancel
      page_controller.start( PG_CLOCK );
      return true;

    case 0: // Set
      // update timebase & sync seconds at zero
      setTime( &tSet );
      page_controller.start( PG_CLOCK );
      return true;

    default: // up/down buttons
      adjustSetTime( &tSet, btn-2, TEXT_COLOR );
  }
  return false; // page was not changed
}

// - - - - - - - - - - - - -
// Alarm Set page
//     HH:MM
//  set   cancel
// - - - - - - - - - - - - -

void alarmSetPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  page_controller.clear(); // prepare to create buttons
  // grab a copy of time
  memcpy( &tSet, &page_controller.clock_page.tAlarm, sizeof(struct ClockTime) );
  // paint time area
  paintTime( TFT_YELLOW, &tSet, SET_TIME_Y );
  // create buttons
  const uint BTN_Y = 190;
  page_controller.createTextButton(   5,BTN_Y,  100,45, "Off",    FSS12 ); // button 0
  page_controller.createTextButton( 110,BTN_Y,  100,45, "Set",    FSS12 ); // button 1
  page_controller.createTextButton( 215,BTN_Y,  100,45, "Cancel", FSS12 ); // button 2
  createSettingButtons(SET_TIME_Y);
  // no alarm while setting alarm
  page_controller.clock_page.alarm_enabled = false;
}

void alarmSetPage::loop()
{
  // time out setting (cancel) & return to normal clock mode
  coBegin
    coDelay( SET_TIMEOUT_MS )
  coEnd
  page_controller.start( PG_CLOCK );
}

boolean alarmSetPage::buttonPress( int btn )
{
  _state_ = 0; // keep alive
  switch (btn)
  { 
    case 2: // Cancel
      page_controller.start( PG_CLOCK );
      return true;

    case 1: // Set
      setAlarm( &tSet );
      page_controller.clock_page.alarm_enabled = true;
      page_controller.start( PG_CLOCK );
      return true;

    case 0: // Alarm off
      page_controller.clock_page.alarm_enabled = false;
      page_controller.start( PG_CLOCK );
      return true;

    default: // up/down buttons
      adjustSetTime( &tSet, btn-3, TFT_YELLOW );
  }
  return false; // page was not changed
}

// - - - - - - - - - - - - -
// Alarm page
//   first minute
//     start flashing
//     low volume alarm sound
//   full alarm
//     loud alarm sound
//   any press is automatic snooze
//     silences sound
//     stops flashing
//     snooze countdown with clock
//     recycle after snooze time
// - - - - - - - - - - - - -

#define MIN_VOLUME 10
#define MAX_VOLUME 20

void alarmPage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextDatum(TC_DATUM); // origin is top center
  // start flashing
  flashTask.control(true);
  flashTextTask(true);
  // start low volume sound
  myDFPlayer.volume(MIN_VOLUME);
  last_t.hours = 99; // force replot
  page_controller.clear(); // no buttons on this page
}

void alarmPage::terminate()
{
  flashTask.control(false);
  myDFPlayer.stop();
  _state_ = 0;
}

void alarmPage::flashTextTask(boolean terminate=false)
{ static int _state_ = 0;  static uint32_t _tm_;
  if (terminate) { _state_ = 0;  return; }
  coBegin
    for (;;)
    { paintTime( TFT_YELLOW, &page_controller.clock_page.tAlarm, ROW2_Y );
      coDelay( 900 )
      tft.fillRect( 0,DISP_H/2, DISP_W,DISP_H/2, BACKGND_COLOR );
      coDelay( 100 )
    }
  coEnd  
}

void alarmPage::loop()
{
  flashTask.loop(); // drive the lamp flasher
  flashTextTask();  // drive the screen text flasher
  // maintain clock display
  struct ClockTime t;
  getTime( &t );  
  if (last_t.hours != t.hours || last_t.minutes != t.minutes)
  { last_t.hours   = t.hours;
    last_t.minutes = t.minutes;
    paintTime( TEXT_COLOR, &t, ROW1_Y );
  }
  // alarm animation sequence
  coBegin
    coWaitWhile( !digitalRead(BUSY_PIN) ) // wait for DFplayer
    myDFPlayer.volume(v = MIN_VOLUME);
    coDelay(50);
    if (mode == 0)
    { // play low volume sound for a minute
      for (i=4; i; i--) // MP3 is 12 seconds => 1 minute
      { coWaitWhile( !digitalRead(BUSY_PIN) ) // wait for DFplayer
        myDFPlayer.play(MP3_BEEP);
        coDelay(50)
      }
    }
    // go full alarm
    for (i=720; i; i--) // MP3 is 5 seconds => 1 hour
    { coWaitWhile( !digitalRead(BUSY_PIN) ) // wait for DFplayer
      myDFPlayer.play(MP3_REDALERT);
      coWaitWhile(!digitalRead(BUSY_PIN) );
      if (v < MAX_VOLUME)
      { v++; // creep volume up
        myDFPlayer.volume(v);
        coDelay(50)
      }
    }
    // abort alarm after an hour
    terminate();
    page_controller.start( PG_CLOCK );
  coEnd
}

// any touch starts snooze mode
boolean alarmPage::rawPress([[maybe_unused]] TS_Point *pt)
{
  // stop sound & lights
  terminate();
  // show snooze page  
  page_controller.start( PG_SNOOZE );
  return true;  
}

// - - - - - - - - - - - - -
// Snooze page
//   time [late]
//   snooze countdown
//   [alarm off]
// - - - - - - - - - - - - -

void snoozePage::init()
{
  tft.fillScreen(BACKGND_COLOR);
  tft.setTextColor( TEXT_COLOR, BACKGND_COLOR );  
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSS18);
  tft.drawString( "Snooze", 230,ROW2_Y);
  // clock in upper half of screen
  struct ClockTime t;
  getTime( &t );  
  minutes_left = SNOOZE_MINUTES;
  last_t.hours   = t.hours;
  last_t.minutes = t.minutes;
  paintTime( TEXT_COLOR, &t, ROW1_Y );
  paintCountdown(); // minutes left
  // disarm alarm button
  page_controller.createTextButton(  15,ROW2_Y+10,  125,75, "Disarm Alarm", FSS12 ); // button 0
  // more snooze (hit anywhere in the snooze display area)
  page_controller.createAreaButton( DISP_W/2,DISP_H/2, DISP_W/2,DISP_H/2 ); // button 1
}

void snoozePage::paintCountdown()
{
  tft.setTextColor( TEXT_COLOR );  
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSS24);
  tft.fillRect( 200,ROW2_Y+40, 60,50, BACKGND_COLOR );  
  tft.drawNumber( minutes_left, 230,ROW2_Y+40);
}

void snoozePage::loop()
{
  // update every minute
  struct ClockTime t;
  getTime( &t );  
  if (last_t.hours != t.hours || last_t.minutes != t.minutes)
  { last_t.hours   = t.hours;
    last_t.minutes = t.minutes;
    minutes_left--;
    if (minutes_left < 0) { page_controller.alarm_page.mode = 1;  page_controller.start( PG_ALARM ); return; }
    paintTime( TEXT_COLOR, &t, ROW1_Y );
    paintCountdown();
  }
}

boolean snoozePage::buttonPress( int btn )
{
  switch (btn)
  { case 0: // alarm off
      page_controller.clock_page.alarm_enabled = false;
      page_controller.start( PG_CLOCK );
      return true;  
    case 1: // more snooze
      init();
  }
  return false; // page was not changed
}

// - - - - - - - - - - - - -
// Touch Screen Calibration
// collect 5 touch samples at each corner of the screen
// - - - - - - - - - - - - -

#define CAL_OFFSET 25 // calibration point radius

void calPage::init()
{
  state = count = 0; // initialize calibration steps
  new_cal.min_x = new_cal.min_y = new_cal.max_x = new_cal.max_y = 0;
  // set default calibration so buttons sorta work
  old_cal = page_controller.touch_cal;
  page_controller.touch_cal = default_cal;
  paint();
}
void calPage::paint()
{
  tft.fillScreen(TFT_BLACK);
  tft.setTextColor( TFT_WHITE, TFT_BLACK );  
  tft.setTextDatum(TC_DATUM); // origin is top center
  tft.setFreeFont(FSS9);
  // first 4 states collect point data
  page_controller.clear(); // clear buttons
  if (state < 4)
  { tft.drawString( "Touch Screen Calibration", DISP_W/2,50); 
    tft.drawString( "Touch the calibration point", DISP_W/2,75); 
    page_controller.clear(); // clear buttons
    page_controller.createTextButton( DISP_W/2-40,DISP_H/2, 80,30, "Cancel", FSS9 ); // btn 0
    tft.fillCircle( screen_x[state], screen_y[state], CAL_OFFSET, TFT_GREEN );
    tft.setTextColor( TFT_BLACK );
    tft.setTextDatum(MC_DATUM);
    tft.drawString( String(5-count), screen_x[state], screen_y[state] );     
  }
  else // 5'th state is test & confirm
  { tft.setTextDatum(TC_DATUM);
    tft.setTextColor( TFT_WHITE );
    tft.drawString( "Calibration Test", DISP_W/2,100); 
    page_controller.createTextButton( DISP_W/2-40,145, 80,50, "Cancel", FSS9 ); // btn 0
    page_controller.createTextButton( DISP_W/2-40, 75, 80,50, "Ok",     FSS9 ); // btn 1
    page_controller.createTextButton( DISP_W/2-40, 15, 80,50, "Retry",  FSS9 ); // btn 2
  }
}
boolean calPage::buttonPress(int btn)
{
  switch (btn)
  { case 1: // ok
      // copy to live
      page_controller.touch_cal.min_x = new_cal.min_x;
      page_controller.touch_cal.min_y = new_cal.min_y;
      page_controller.touch_cal.max_x = new_cal.max_x;
      page_controller.touch_cal.max_y = new_cal.max_y;
      // save to EEPROM
      EEPROM.put( 0, page_controller.touch_cal );
      EEPROM.commit();
      page_controller.start(PG_CLOCK);
      break;      
    case 0: // cancel
      page_controller.touch_cal = old_cal; // restore original calibration
      page_controller.start(PG_CLOCK);
      break;
    case 2: // retry
      state = count = 0; // initialize calibration steps
      new_cal.min_x = new_cal.min_y = new_cal.max_x = new_cal.max_y = 0;
      paint();
  }
  return true;
}
boolean calPage::rawPress(TS_Point *pt)
{
  if (state < 4) // collecting data
  { switch (state)
    { case 0: // upper left
        if (pt->x > 1500 || pt->y > 1500) return false; // sanity check
        new_cal.min_x += pt->x;  new_cal.min_y += pt->y;  break;
      case 1: // upper right
        if (pt->x < 2600 || pt->y > 1500) return false; // sanity check
        new_cal.max_x += pt->x;  new_cal.min_y += pt->y;  break;
      case 2: // lower left
        if (pt->x > 1500 || pt->y < 2600) return false; // sanity check
        new_cal.min_x += pt->x;  new_cal.max_y += pt->y;  break;
      case 3: // lower right
        if (pt->x < 2600 || pt->y < 2600) return false; // sanity check
        new_cal.max_x += pt->x;  new_cal.max_y += pt->y;  break;
    }
    if (++count > 4)
    { count = 0;
      if (++state >= 4) // data collection done
      { // calc averages
        new_cal.min_x /= 10;
        new_cal.max_x /= 10;
        new_cal.min_y /= 10;
        new_cal.max_y /= 10;
        
        // map origin & span
        int min = new_cal.min_x;
        int max = new_cal.max_x;
        new_cal.min_x = map(   0,      CAL_OFFSET, DISP_W-CAL_OFFSET, min,max );
        new_cal.max_x = map( DISP_W-1, CAL_OFFSET, DISP_W-CAL_OFFSET, min,max );
        min = new_cal.min_y;
        max = new_cal.max_y;
        new_cal.min_y = map(   0,      CAL_OFFSET, DISP_H-CAL_OFFSET, min,max );
        new_cal.max_y = map( DISP_H-1, CAL_OFFSET, DISP_H-CAL_OFFSET, min,max );
      }
    }      
    paint();
  }
  else // test
  { // convert point to screen coordinates using new calibration data
    TS_Point pt_screen;
    pt_screen.x = map(pt->x, new_cal.min_x, new_cal.max_x, 0, tft.width() );
    pt_screen.y = map(pt->y, new_cal.min_y, new_cal.max_y, 0, tft.height());
    tft.drawCircle( pt_screen.x, pt_screen.y, CAL_OFFSET, TFT_YELLOW );
  }
  return false; // eat raw press, attempt to find button press
}
