//------------------------------------------------------
// Screens for Touch Clock

#pragma once
#ifndef Lib_Pages
#define Lib_Pages

#include "app.h"

#include "TFT_eSPI.h"
#include "Free_Fonts.h"
extern TFT_eSPI tft;

#include <XPT2046_Touchscreen.h>
extern XPT2046_Touchscreen ts;

#include <DFPlayerMini_Fast.h> // https://github.com/PowerBroker2/DFPlayerMini_Fast
extern DFPlayerMini_Fast myDFPlayer;

#define BACKGND_COLOR TFT_BLACK
#define TEXT_COLOR    TFT_WHITE

#define BUTTON_COLOR  tft.color565(64,64,64) //DARKDARKGRAY 
#define BUTTON_HILITE TFT_ORANGE
#define BUTTON_TEXT   TFT_YELLOW
#define BUTTON_BORDER TFT_WHITE

//------------------------------------------------------
// Page base class

class Page
{
  public:
    virtual void loop() { ; }
    virtual void init() { ; }
    virtual boolean buttonPress([[maybe_unused]] int btn) { return false; } // returns true if page changed
    virtual boolean rawPress([[maybe_unused]] TS_Point *pt) { return false; } // returns true if press was handled
};

//------------------------------------------------------
// Pages

// - - - - - - - - - - - - -
// Splash
class splashPage : public Page
{
  private:
    int _state_ = 0; // coroutine state
    uint32_t _tm_;   // for delay functions

  public:
    void init();
    void loop();
    boolean rawPress( TS_Point *pt );
};

// - - - - - - - - - - - - -
// Clock
class clockPage : public Page
{
  private:
    struct ClockTime last_t;
    
  public:
    struct ClockTime tAlarm;
    boolean alarm_enabled;
    void init();
    void loop();
    boolean buttonPress(int btn);
};

// - - - - - - - - - - - - -
// Clock Set
class clockSetPage : public Page
{
  private:
    struct ClockTime tSet; // time being set
    int _state_ = 0; // coroutine state
    uint32_t _tm_;   // for delay functions

  public:
    void init();
    void loop();
    boolean buttonPress(int btn);
};

// - - - - - - - - - - - - -
// Alarm Set
class alarmSetPage : public Page
{
  private:
    struct ClockTime tSet; // time being set
    int _state_ = 0; // coroutine state
    uint32_t _tm_;   // for delay functions

  public:
    void init();
    void loop();
    boolean buttonPress(int btn);
};

// - - - - - - - - - - - - -
// Alarm
class alarmPage : public Page
{
  private:
    struct ClockTime last_t;
    uint i,v;
    int _state_ = 0; // coroutine state
    uint32_t _tm_;   // for delay functions
    
  public:
    uint mode;       // 0=startup, 1=snooze resume
    void init();
    void loop();
    void terminate();
    void flashTextTask(boolean terminate);
    boolean rawPress(TS_Point *pt);
};

// - - - - - - - - - - - - -
// Snooze
class snoozePage : public Page
{
  private:
    int    minutes_left;
    struct ClockTime last_t;
    
  public:
    void init();
    void loop();
    boolean buttonPress(int btn);
    void paintCountdown();
};

// - - - - - - - - - - - - -
// Calibration
class calPage : public Page
{
  public:
    void init();
    boolean buttonPress(int btn);
    boolean rawPress(TS_Point *pt);
  private:
    void paint();
    uint state;
    uint count;
    const int16_t screen_x[4] = { 50, 319-50, 50, 319-50 }; // landscape
    const int16_t screen_y[4] = { 50, 50, 249-50, 249-50 };
    touch_cal_struct old_cal;
    touch_cal_struct default_cal = { 542,3560,887,3651 }; // landscape
    struct 
    { int32_t min_x; 
      int32_t max_x;
      int32_t min_y;
      int32_t max_y;   
    } new_cal;  
};

//------------------------------------------------------
//-- Page Management -----------------------------------
//------------------------------------------------------

#define PG_SPLASH   0
#define PG_CLOCK    1
#define PG_CAL      2
#define PG_ALARM    3
#define PG_SETCLOCK 4
#define PG_SETALARM 5
#define PG_SNOOZE   6
#define NUM_PAGES   7

#define MAX_BUTTONS 15
#define TOUCH_DEBOUNCE_MSEC 100

class PageController
{
  private:
    #define BTN_TEXT  0 // text button
    #define BTN_AREA  1 // invisible button

    uint button_count = 0;
    struct
    { uint8_t  type;          // BTN_x
      const    GFXfont *font;
      int16_t  x,y;           // display box area
      int16_t  w,h;
      String   text;
      uint8_t  group;         // group button ID
      boolean  selected;      // group button state
      int16_t  hit_x,hit_y;   // hit box area
      int16_t  hit_w,hit_h;
    } button[ MAX_BUTTONS ];

    // button hiliting
    #define  HILITE_MSEC 500
    uint     button_number = 0; // button being pressed now
    uint32_t button_tm;         // hilite timer

  //-- scan for button hit

    uint distance2( TS_Point *pt1, uint pt2_x, uint pt2_y )
    {
      int dx = pt1->x - pt2_x;
      int dy = pt1->y - pt2_y;
      return pow(abs(dx),2) + pow(abs(dy),2);
    }

    #define BUTTON_HIT_MARGIN 20 // pixels
    int scanButtonTable( TS_Point *pt ) 
    {
      if (button_count == 0) return -1;
      // find button closest to touch point
      uint best_i       = 0;
      uint best_dist_sq = distance2( pt, button[0].x + button[0].w/2, button[0].y + button[0].h/2 );
      for (uint i=1; i < button_count; i++)
      { uint dist_sq = distance2( pt, button[i].x + button[i].w/2, button[i].y + button[i].h/2 );
        if (dist_sq < best_dist_sq) { best_i = i;  best_dist_sq = dist_sq; }
      }
      // is the hit actually within button bounds?
      if (pt->x >= button[best_i].hit_x-BUTTON_HIT_MARGIN && pt->x <= button[best_i].hit_x+button[best_i].hit_w+BUTTON_HIT_MARGIN
       && pt->y >= button[best_i].hit_y-BUTTON_HIT_MARGIN && pt->y <= button[best_i].hit_y+button[best_i].hit_h+BUTTON_HIT_MARGIN)
         return best_i;
      // button not hit
      return -1; 
    }

  //-- touch input

    boolean  is_touched = false;
    uint32_t touch_tm;
    boolean touchInput( TS_Point *pt )
    {
      if (ts.touched())
      { touch_tm = millis(); // [re]start debounce
        if (!is_touched)
        { // touch event
          *pt = ts.getPoint();
           is_touched = true;
           touch_tm = millis();
           return true;
        }
      }
      else // not touched
        if (is_touched)
          if (millis() - touch_tm > TOUCH_DEBOUNCE_MSEC)
            is_touched = false;  
    
      return false;
    }

  //-- button management

  public:
      
    void clear() { button_count = 0; }

    void drawButton( uint bi, boolean hilite = false )
    {
      if (bi >= button_count) {Serial.println("FUCK YOU"); return;}
//tft.drawRect( button[bi].x,button[bi].y,button[bi].w,button[bi].h, TFT_MAGENTA );
      button[bi].selected = hilite;
      switch (button[bi].type)
      { case BTN_TEXT:
          tft.fillRoundRect( button[bi].x,button[bi].y,button[bi].w,button[bi].h,8,hilite ? BUTTON_HILITE:BUTTON_COLOR );
          tft.drawRoundRect( button[bi].x,button[bi].y,button[bi].w,button[bi].h,8,BUTTON_BORDER);
          tft.setTextColor( hilite ? BACKGND_COLOR:BUTTON_TEXT );  
          tft.setFreeFont(button[bi].font);
          tft.setTextDatum(MC_DATUM); // origin is middle center
          { int i = button[bi].text.indexOf(' ');
            if (i == -1)
              // single line
              tft.drawString( button[bi].text, button[bi].x+button[bi].w/2-2, button[bi].y+button[bi].h/2-2);
            else // double line
            { uint LS = tft.fontHeight();
              String s1 = button[bi].text.substring(0,i);
              String s2 = button[bi].text.substring(i+1);
              tft.drawString( s1, button[bi].x+button[bi].w/2-2, button[bi].y+button[bi].h/2-2-LS/2);
              tft.drawString( s2, button[bi].x+button[bi].w/2-2, button[bi].y+button[bi].h/2-2+LS/2);
            } 
          }
      }
    }
    boolean allocateButton( int x, int y, int w, int h, int type )
    {
      if (button_count >= MAX_BUTTONS) {Serial.println("ALLOCATE FAILED"); return false;}
      button[button_count].type = type;
      button[button_count].x = button[button_count].hit_x = x;
      button[button_count].y = button[button_count].hit_y = y;
      button[button_count].w = button[button_count].hit_w = w;
      button[button_count].h = button[button_count].hit_h = h;
      button_count++;
      return true;
    }
    void createTextButton( int x, int y, int w, int h, String text, const GFXfont *font )
    {
      if (!allocateButton( x,y,w,h,BTN_TEXT )) return;
      button[button_count-1].text = text;
      button[button_count-1].font = font;
      drawButton( button_count-1 );
    }
    void createAreaButton( int x, int y, int w, int h )
    {
      if (!allocateButton( x,y,w,h,BTN_AREA )) return;
      drawButton( button_count-1 );
    }

  //-- page management
  
  public:

    // touch screen calibration stored in emulated EEPROM
    touch_cal_struct touch_cal; 

    // show a page
    void start(uint page_index) 
    {
      if (page_index >= NUM_PAGES) return; // failsafe
      // show the screen      
      ui_page[current_page = page_index]->init();
    }
    
    void loop() 
    {
      // drive the current page
      ui_page[current_page]->loop();
      
      // watch for touch input
      TS_Point pt_raw, pt_screen;
      if (touchInput( &pt_raw ))
      {
//Serial.print("TOUCH "); Serial.print(pt_raw.x); Serial.print(","); Serial.println(pt_raw.y);  
        // scale to screen coordinates using calibration
        pt_screen.x = map(pt_raw.x, touch_cal.min_x, touch_cal.max_x, 0, DISP_W );
        pt_screen.y = map(pt_raw.y, touch_cal.min_y, touch_cal.max_y, 0, DISP_H);
//Serial.print(pt_screen.x); Serial.print(","); Serial.println(pt_screen.y);
        pt_screen.x = constrain(pt_screen.x, 0,DISP_W-1);
        pt_screen.y = constrain(pt_screen.y, 0,DISP_H-1);
//tft.drawCircle( pt_screen.x, pt_screen.y, 25, TFT_YELLOW );        
        // scan for button hit
        int btn = scanButtonTable(&pt_screen);
        if (btn >= 0) // send button to page
        { switch (button[btn].type)
          { 
            case BTN_TEXT:
              // start button hilite animation
              drawButton(btn,true); // hilite
              button_number = btn+1;
              button_tm = millis();
              break;
              
            case BTN_AREA:
              // send button to page
              ui_page[current_page]->buttonPress(btn);
              break;
          }
        }
        else // button not hit, send raw coords
          if (button_number == 0) // not if button is in progress
            ui_page[current_page]->rawPress(&pt_raw);
      }
      // button animation
      if (button_number > 0)
        if (millis() - button_tm > HILITE_MSEC)
        { // okay now send button to page
          if (!ui_page[current_page]->buttonPress(button_number-1))
            drawButton(button_number-1); // didn't leave page, un-hilite button
          button_number = 0;
        }
    }
    
    // create the pages
    splashPage   splash_page;
    clockPage    clock_page;
    calPage      cal_page;
    alarmPage    alarm_page;
    alarmSetPage alarm_set_page;
    clockSetPage clock_set_page;
    snoozePage   snooze_page;
    // make a table of pages
    Page *ui_page[NUM_PAGES] = 
    { &splash_page, &clock_page, &cal_page, &alarm_page, &clock_set_page, &alarm_set_page, &snooze_page };
    uint current_page = PG_SPLASH;
};

#endif
